unit lrgunit;
{ This unit implements a large integer type which can represent        }
{ integers of up to DigitLis.MaxDigits digits.  These large integers   }
{ can be manipulated using functions and procedures specified and      }
{ defined in this unit                                                 }

{ Each large integer, represented by the type LargeInt, represents }
{ a valid integer or an error specified by ErrorType.  If an arithmetic}
{ operation, e.g., addition, is performed on a large integer}
{ which is in an error state (not valid), then the resulting large integer}
{ will have the error status DerivedError.  The error status Overflow can}
{ result from multiplying two large integers so that the product exceeds}
{ MaxDigits digits.  The BadInput error results when an invalid integer is}
{ read, e.g., 123a456}

interface

uses
  digunit;

type
  SignType = (Negative, Positive);
  ErrorType = (NoError, Overflow, BadInput, DerivedError);
  LargeInt = record
    Sign: SignType;
    Digits: DigitListType;
    Error: ErrorType;
  end;

procedure LargeConstruct (var Large: LargeInt);
{ Pre: Large is uninitialized}
{ Post: Large is initialized, and has a positive sign, }
{       an error status of NoError and a value of 0 }

procedure LargeDestroy (var Large: LargeInt);
{ Pre: Large is initialized}
{ Post: Large is uninitialized }

procedure ErrorWrite (var outfile: text; E: ErrorType);
{ Pre: outfile is open for writing}
{ Post: Error message corresponding to E has been written to outfile. }

procedure LargeReadln (var infile: text; var Large: LargeInt);
{ Pre: infile is open for reading, Large is initialized                }
{ Post: infile has been read through EOLN; Large represents the integer}
{         read from infile, or has the error status BadInput           }
{         A valid integer consists of a sequence of digits of          }
{         length <= MaxDigits, optionally preceded by a single '+'     }
{         or '-' sign. }

procedure LargeWrite (var outfile: text; var Large: LargeInt);
{ Pre: outfile is open for writing; Large is initialized        }
{ Post: The digits of Large are printed to the file outfile;    }
{       if Large is negative, a leading '-' is printed.         }
{       If Large is invalid, i.e., has an error status other    }
{       than NoError, then an error message is printed instead. }

function ErrorOf (var Large: LargeInt): ErrorType;
{ Pre: Large is initialized}
{ Post: returns the error status of Large }

function LargeEqual (var A, B: LargeInt): boolean;
{ Pre: A and B are initialized}
{ Post: returns true if A = B, otherwise returns false }

function LargeGreaterThan (var A, B: LargeInt): boolean;
{ Pre: A and B are initialized}
{ Post: returns true if A > B, otherwise returns false }

function LargeLessThan (var A, B: LargeInt): boolean;
{ Pre: A and B are initialized}
{ Post: returns true if A < B, otherwise returns false }

procedure LargeAssign (var A: LargeInt; var B: LargeInt);
{ Pre: A and B are initialized}
{ Post: A has value held by B, i.e., effects A := B }

procedure LargeSubtract (var A, B: LargeInt; var Diff: LargeInt);
{ Pre: A, B and Diff are initialized                                 }
{ Post: Diff contains the arithmetic difference A - B, or an Overflow}
{       Error status if this difference exceeds MaxDigits digits     }
{       or a Derived Error status if either A or B are invalid       }

procedure LargeAdd (var A, B: LargeInt; var Sum: LargeInt);
{ Pre: A, B and Sum are initialized                                    }
{ Post: Sum contains the arithmetic sum of A and B, or an Overflow     }
{       Error status if this sum exceeds MaxDigits digits or a Derived }
{       Error status if either A or B are invalid                      }

procedure SmallMultiply (X: integer; var A: LargeInt; var B: LargeInt);
{ Pre: A and B are initialized, 0 <= X < 10                            }
{ Post: B contains the product  X*A , an Overflow Error status if      }
{       this product exceeds MaxDigits digits or  a DerivedError status}
{       if A contains an error or if X is out of range                 }

procedure LargeMultiply (var A, B: LargeInt; var Product: LargeInt);
{ Pre: A, B and Product are initialized                                   }
{ Post: Product contains the arithmetic product of A and B, or an Overflow}
{       Error status if this product exceeds MaxDigits digits or          }
{       a Derived Error status if either A or B are invalid               }

procedure IntToLarge (N: integer; var Large: LargeInt);
{ Pre: Large is initialized}
{ Post: Large has value of N }

procedure LargeToInt (var Large: LargeInt; var N: integer);
{ Pre: Large is initialized, -MaxInt < Large < +MaxInt}
{ Post: N has value of Large }

procedure LargeToReal (var Large: LargeInt; var R: real);
{ Pre: Large is initialized and represents a value less than}
{      the maximum real value}
{ Post: R has value of Large }

implementation

procedure LargeConstruct (var Large: LargeInt);
{ Pre: Large is uninitialized}
{ Post: Large is initialized, and has a positive sign, }
{       an error status of NoError and a value of 0 }

begin
  Large.Error := NoError;
  DigitListConstruct(Large.Digits);
  AddLeftDigit(Large.Digits, 0);
  Large.Sign := Positive;
end;

procedure LargeDestroy (var Large: LargeInt);
{ Pre: Large is initialized}
{ Post: Large is uninitialized }

begin
  DigitListDestroy(Large.Digits);
end;

procedure ErrorWrite (var outfile: text; E: ErrorType);
{ Pre: outfile is open for writing}
{ Post: Error message corresponding to E has been written to outfile. }

begin
  case E of
    NoError:
      write(outfile, 'No Error');
    Overflow:
      write(outfile, 'Overflow');
    BadInput:
      write(outfile, 'Bad Input');
    DerivedError:
      write(outfile, 'Derived Error');
  end;
end;

procedure TrimZeroes (var Large: LargeInt);
{ Pre: Large holds a sequence of digits possibly with leading zeroes}
{ Post: Large is returned with all leading zeroes removed (zero left as 0) }

var
  Done: boolean;

begin
  ResetLeft(Large.Digits);
  Done := false;
  while not Done do
  begin
    if EOListRight(Large.Digits) then
      Done := true
    else if CurrDigit(Large.Digits) <> 0 then
      Done := true
    else
    begin
      RemoveLeftDigit(Large.Digits);
      ResetLeft(Large.Digits);
    end;
  end;
  if DigitListLength(Large.Digits) = 0 then
  begin
    AddLeftDigit(Large.Digits, 0);
    Large.Sign := Positive;
  end;
end;


procedure LargeReadln (var infile: text; var Large: LargeInt);
{ Pre: infile is open for reading, Large is initialized                }
{ Post: infile has been read through EOLN; Large represents the integer}
{         read from infile, or has the error status BadInput           }
{         A valid integer consists of a sequence of digits of          }
{         length <= MaxDigits, optionally preceded by a single '+'     }
{         or '-' sign. }

var
  FirstChar: boolean;
  ch: char;

begin
  Large.Error := NoError;
  Large.Sign := Positive;
  FirstChar := true;
  DigitListClear(Large.Digits);
  with Large do
  while not eoln(infile) do
  begin
    read(infile, ch);
    if FirstChar and (ch = '+') then
      sign := Positive
    else if FirstChar and (ch = '-') then
      sign := Negative
    else if ch in ['0'..'9'] then
      if DigitListLength(Digits) < MaxDigits then
	AddRightDigit(Digits, ord(ch) - ord('0'))
      else
	Error := Overflow
    else
      Error := BadInput;
    FirstChar := false;
  end;
  readln(infile);
  TrimZeroes(Large);
end;

procedure LargeWrite (var outfile: text; var Large: LargeInt);
{ Pre: outfile is open for writing; Large is initialized        }
{ Post: The digits of Large are printed to the file outfile;    }
{       if Large is negative, a leading '-' is printed.         }
{       If Large is invalid, i.e., has an error status other    }
{       than NoError, then an error message is printed instead. }

var
  ColumnsToPrint, DigitPos: integer;

begin
  if Large.Error = NoError then
  begin
    if Large.sign = Negative then
      write(outfile, '-');
    ResetLeft(Large.Digits);
    while not EOListRight(Large.Digits) do
    begin
      write(outfile, CurrDigit(Large.Digits) : 1);
      MoveRight(Large.Digits);
    end;
  end
  else
    ErrorWrite(outfile, Large.Error);
end;

function ErrorOf (var Large: LargeInt): ErrorType;
{ Pre: Large is initialized}
{ Post: returns the error status of Large }

begin
  ErrorOf := Large.Error;
end;

function LargeEqual (var A, B: LargeInt): boolean;
{ Pre: A and B are initialized}
{ Post: returns true if A = B, otherwise returns false }

var
  Equal: boolean;

begin
  if A.Sign <> B.Sign then
    LargeEqual := false
  else if DigitListLength(A.Digits) <> DigitListLength(B.Digits) then
    LargeEqual := false
  else
  begin  { Same sign and length }
    ResetRight(A.Digits);
    ResetRight(B.Digits);
    Equal := true;
    while Equal and not EOListLeft(A.Digits) do
    begin
      if CurrDigit(A.Digits) = CurrDigit(B.Digits) then
      begin
	MoveLeft(A.Digits);
	MoveLeft(B.Digits);
      end
      else
	Equal := false;
    end;
    LargeEqual := Equal;
  end;
end;


function LargeGreaterThan (var A, B: LargeInt): boolean;
{ Pre: A and B are initialized}
{ Post: returns true if A > B, otherwise returns false }

var
  Equal: Boolean;

begin
  if A.Sign <> B.Sign then
    LargeGreaterThan := (A.Sign = Positive)
  else { Signs are the same }
  if DigitListLength(A.Digits) > DigitListLength(B.Digits) then
    LargeGreaterThan := (A.Sign = Positive)
  else if DigitListLength(A.Digits) < DigitListLength(B.Digits) then
    LargeGreaterThan := (A.Sign = Negative)
  else { Same length, same sign }
  begin
    ResetLeft(A.Digits);
    ResetLeft(B.Digits);
    Equal := true;
    while Equal and not EOListRight(A.Digits) do
    begin
      if CurrDigit(A.Digits) = CurrDigit(B.Digits) then
      begin
	MoveRight(A.Digits);
	MoveRight(B.Digits);
      end
      else
	Equal := false;
    end;
    if Equal then
      LargeGreaterThan := false  { since must be equal }
    else if CurrDigit(A.Digits) > CurrDigit(B.Digits) then
      LargeGreaterThan := (A.Sign = Positive)
    else
      LargeGreaterThan := (A.Sign = Negative);
  end;
end;

function LargeLessThan (var A, B: LargeInt): boolean;
{ Pre: A and B are initialized}
{ Post: returns true if A < B, otherwise returns false }

begin
  LargeLessThan := LargeGreaterThan(B, A);
end;

procedure LargeAssign (var A: LargeInt; var B: LargeInt);
{ Pre: A and B are initialized}
{ Post: A has value held by B, i.e., effects A := B }

begin
  DigitListAssign(A.Digits, B.Digits);
  A.Sign := B.Sign;
  A.Error := B.Error;
end;

procedure LargeSubtract;{var A,B: LargeInt; var Diff: LargeInt}
{ Pre: A, B and Diff are initialized                                 }
{ Post: Diff contains the arithmetic difference A - B, or an Overflow}
{       Error status if this difference exceeds MaxDigits digits     }
{       or a Derived Error status if either A or B are invalid       }

var
  ADigit: -1..19;   { -1 needed in case borrow from 0 }
  BDigit, DigitDiff: 0..9;
  Borrow: 0..1;
  CopyA, CopyB: LargeInt;

begin
  LargeConstruct(CopyA);
  LargeAssign(CopyA, A);
  LargeConstruct(CopyB);
  LargeAssign(CopyB, B);
  if (CopyA.Error <> NoError) or (CopyB.Error <> NoError) then
    Diff.Error := DerivedError
  else if (CopyA.Sign <> CopyB.Sign) then
  begin    { Use the fact that x - y = x + (-y) }
    CopyB.Sign := CopyA.Sign;
    LargeAdd(CopyA, CopyB, Diff);
  end
  else { Signs must be equal }
  begin
    if (LargeGreaterThan(CopyB, CopyA) and (CopyA.Sign = Positive))
       or (LargeGreaterThan(CopyA, CopyB) and (CopyA.Sign = Negative)) then
    begin { Use the fact that x - y  =  -(y - x) }
      LargeSubtract(CopyB, CopyA, Diff);
      if Diff.Sign = Negative then
	Diff.Sign := Positive
      else
	Diff.Sign := Negative;
    end
    else  { same sign,  Abs(A) > Abs(B) }
    begin
      Diff.Sign := CopyA.Sign;
      Diff.Error := NoError;
      DigitListClear(Diff.Digits);
      ResetRight(CopyA.Digits);
      ResetRight(CopyB.Digits);
      Borrow := 0;
      while not EOListLeft(CopyA.Digits) do
      begin
	ADigit := CurrDigit(CopyA.Digits) - Borrow;
	if not EOListLeft(CopyB.Digits) then
	  BDigit := CurrDigit(CopyB.Digits)
	else
	  BDigit := 0;
	Borrow := 0;
	if Adigit < BDigit then
	begin
	  Adigit := 10 + ADigit;
	  Borrow := 1;
	end;
	DigitDiff := ADigit - BDigit;
	AddLeftDigit(Diff.Digits, DigitDiff);
	MoveLeft(CopyA.Digits);
	MoveLeft(CopyB.Digits);
      end; { while }
      TrimZeroes(Diff);
    end;
  end;
  LargeDestroy(CopyA);
  LargeDestroy(CopyB);
end;


procedure LargeAdd (var A, B: LargeInt; var Sum: LargeInt);
{ Pre: A, B and Sum are initialized                                    }
{ Post: Sum contains the arithmetic sum of A and B, or an Overflow     }
{       Error status if this sum exceeds MaxDigits digits or a Derived }
{       Error status if either A or B are invalid                      }

var
  DigitSum: 0..19;
  ADigit, BDigit: 0..9;
  Carry: 0..1;
  CopyA, CopyB: LargeInt;

begin
  LargeConstruct(CopyA);
  LargeAssign(CopyA, A);
  LargeConstruct(CopyB);
  LargeAssign(CopyB, B);
  if (CopyA.Error <> NoError) or (CopyB.Error <> NoError) then
    Sum.Error := DerivedError
  else if CopyA.Sign <> CopyB.Sign then
  begin  { Use the fact that A + B = A - (-B), so change B's sign }
    if CopyB.Sign = Positive then
      CopyB.Sign := Negative
    else
      CopyB.Sign := Positive;
      LargeSubtract(CopyA, CopyB, Sum);
  end
  else { Signs are equal }
  begin
    Sum.Error := NoError;
    Sum.Sign := CopyA.Sign;
    DigitListClear(Sum.Digits);
    Carry := 0;
    ResetRight(CopyA.Digits);
    ResetRight(CopyB.Digits);
    while not (EOListLeft(CopyA.Digits) and EOListLeft(CopyB.Digits)) do
    begin
      if not EOListLeft(CopyA.Digits) then
	ADigit := CurrDigit(CopyA.Digits)
      else
	ADigit := 0;
      if not EOListLeft(CopyB.Digits) then
	BDigit := CurrDigit(CopyB.Digits)
      else
	BDigit := 0;
      DigitSum := Carry + ADigit + BDigit;
      AddLeftDigit(Sum.Digits, DigitSum mod 10);
      Carry := DigitSum div 10;
      if not EOListLeft(CopyA.Digits) then
	MoveLeft(CopyA.Digits);
      if not EOListLeft(CopyB.Digits) then
	MoveLeft(CopyB.Digits);
    end;
    if Carry <> 0 then
      if DigitListLength(Sum.Digits) < MaxDigits then
	AddLeftDigit(Sum.Digits, Carry)
      else
	Sum.Error := Overflow;
  end;
  LargeDestroy(CopyA);
  LargeDestroy(CopyB);
end;

procedure SmallMultiply (X: integer; var A: LargeInt; var B: LargeInt);
{ Pre: A and B are initialized, 0 <= X < 10                            }
{ Post: B contains the product  X*A , an Overflow Error status if      }
{       this product exceeds MaxDigits digits or a DerivedError status }
{       if A contains an error or if X is out of range                 }

var
  Carry: Integer;
  DigitProduct: Integer;
  CopyA: LargeInt;

begin
  LargeConstruct(CopyA);
  LargeAssign(CopyA, A);
  if CopyA.Error <> NoError then
    B.Error := DerivedError
  else if (X < 0) or (10 <= X) then
    B.Error := DerivedError        {not really derived error, but it will do}
  else
  begin
    B.Error := CopyA.Error;
    Carry := 0;
    DigitListClear(B.Digits);
    B.Sign := CopyA.Sign;
    ResetRight(CopyA.Digits);
    while not EOListLeft(CopyA.Digits) do
    begin
      DigitProduct := X * CurrDigit(CopyA.Digits) + Carry;
      Carry := DigitProduct div 10;
      if DigitListLength(B.Digits) < MaxDigits then
	AddLeftDigit(B.Digits, DigitProduct mod 10)
      else
	B.Error := Overflow;
      MoveLeft(CopyA.Digits);
    end;

    while Carry > 0 do
    begin
      if DigitListLength(B.Digits) < MaxDigits then
	AddLeftDigit(B.Digits, Carry mod 10)
      else
	B.Error := Overflow;
      Carry := Carry div 10;
    end;
  end;
  LargeDestroy(CopyA);
end;

procedure LargeMultiply (var A, B: LargeInt; var Product: LargeInt);
{ Pre: A, B and Product are initialized                                   }
{ Post: Product contains the arithmetic product of A and B, or an Overflow}
{       Error status if this product exceeds MaxDigits digits or          }
{       a Derived Error status if either A or B are invalid               }

var
  Place, Shift: integer;
  Partial: LargeInt;
  CopyA, CopyB: LargeInt;

begin
  if (A.Error <> NoError) or (B.Error <> NoError) then
    Product.Error := DerivedError
  else
  begin
    LargeConstruct(CopyA);
    LargeAssign(CopyA, A);
    LargeConstruct(CopyB);
    LargeAssign(CopyB, B);
    DigitListClear(Product.Digits);
    AddLeftDigit(Product.Digits, 0);
    Product.Sign := Positive;
    Product.Error := NoError;
    ResetRight(CopyB.Digits);
    Place := 1;
    while not EOListLeft(CopyB.Digits) do
    begin
      SmallMultiply(CurrDigit(CopyB.Digits), CopyA, Partial);
      for Shift := 2 to Place do
	if DigitListLength(Partial.Digits) < MaxDigits then
	  AddRightDigit(Partial.Digits, 0)
	else
	  Product.Error := Overflow;
      LargeAdd(Partial, Product, Product);
      MoveLeft(CopyB.Digits);
      Place := Place + 1;
    end;
    if CopyA.Sign <> CopyB.Sign then
      Product.Sign := Negative
    else
      Product.Sign := Positive;
    TrimZeroes(Product);
    LargeDestroy(CopyA);
    LargeDestroy(CopyB);
  end;
end;

procedure IntToLarge (N: integer; var Large: LargeInt);
{ Pre: Large is initialized}
{ Post: Large has value held by N }

begin
  if N < 0 then
    Large.Sign := Negative
  else
    Large.Sign := Positive;
  N := Abs(N);
  Large.Error := NoError;
  DigitListClear(Large.Digits);
  repeat
    AddLeftDigit(Large.Digits, N mod 10);
    N := N div 10;
  until N = 0;
end;

procedure LargeToInt (var Large: LargeInt; var N: integer);
{ Pre: Large is initialized, -MaxInt < Large < +MaxInt}
{ Post: N has value held by Large }

begin
  N := 0;
  ResetLeft(Large.Digits);
  while not EOListRight(Large.Digits) do
  begin
    if (MaxInt - CurrDigit(Large.Digits)) div 10 >= N then
      N := N * 10 + CurrDigit(Large.Digits)
    else
      writeln('error: maxint exceeded in LargeToInt');
    MoveRight(Large.Digits);
  end;
  if Large.Sign = Negative then
    N := -N;
end;

procedure LargeToReal (var Large: LargeInt; var R: real);
{ Pre: Large is initialized and represents a value less than }
{      the maximum real value}
{ Post: R has value held by Large as a real }

begin
  R := 0;
  ResetLeft(Large.Digits);
  while not EOListRight(Large.Digits) do
  begin
    R := R * 10 + CurrDigit(Large.Digits);
    MoveRight(Large.Digits);
  end;
  if Large.Sign = Negative then
    R := -R;
end;


end. { unit LargeInt }