// EZTWAIN.C - Easy TWAIN DLL, main module
//
// 	05/11/94	spike	initial version
// 1.00	06/17/94	spike
// 1.01	07/27/94	spike - added XferCount negotiation
// 1.02 10/26/94	spike - fixed bug in ModalEventLoop (thanks to Bill DeVoe of HP)
//							replaced random flags with state tracking
// 1.03 02/06/95	spike - fixed gross omissions in WriteDibToFile
// 1.04 04/05/95	spike - added WriteNativeToFile, WriteNativeToFilename, FreeNative
//							added Get/SetCurrentUnits, SetCurrentPixelType
//							added SetCurrentResolution, SetCapOneValue
//							corrected bug in SelectImageSource

#define VERSION 104				// version number, times 100.

//------------ Includes

#include <assert.h>
#include <memory.h>		// memset

#include "windows.h"
#include "commdlg.h"
#include "twain.h"
#include "eztwain.h"

//------------ Constants and Macros

#define STATIC static
#define VALID_HANDLE 32
#define IsValidHandle(h) ((h!=0) && ((h) >= 32))

typedef enum {
	ED_NONE,
	ED_START_TRIPLET_ERRS,
	ED_CAP_GET,				// MSG_GET triplet on a capability failed
	ED_CAP_SET,				// MSG_SET triplet on capability failed
	ED_DSM_FAILURE,			// TWAIN DSM returned TWRC_FAILURE
	ED_DS_FAILURE,			// source returned TWRC_FAILURE
	ED_END_TRIPLET_ERRS,
	ED_NOT_STATE_4,			// operation invoked in wrong state
	ED_NULL_HCON,			// MSG_GET returned a null container handle
	ED_BAD_HCON,			// MSG_GET returned an invalid/unlockable container handle
	ED_BAD_CONTYPE,			// returned container ConType is not valid.
	ED_BAD_ITEMTYPE,		// returned container ItemType is not valid.
	ED_CAP_GET_EMPTY,		// returned container has 0 items.
	ED_CAP_SET_EMPTY,		// trying to restrict a cap to empty set
} ErrorDetail;

const char *pszErrDescrip[] = 
	{	"[no details available]",
		"",
		"DAT_CAPABILITY/MSG_GET failed",
		"DAT_CAPABILITY/MSG_SET failed",
		"Source Manager operation failed",
		"DataSource operation failed",
		"",
		"TWAIN session not in State 4 (Source Open)",
		"MSG_GET returned a NULL container handle",
		"MSG_GET returned an invalid container handle",
		"Returned container is not valid type",
		"Returned container has invalid ItemType",
		"Returned container is empty",
		"App and source found NO values in common",
	};

const char *pszRC[] = {
	"TWRC_SUCCESS",
	"TWRC_FAILURE",
	"TWRC_CHECKSTATUS ('tried hard')",
	"TWRC_CANCEL",
	"TWRC_DSEVENT",
	"TWRC_NOTDSEVENT",
	"TWRC_XFERDONE",
	"TWRC_ENDOFLIST",
};

const char *pszCC[] = {
	"TWCC_SUCCESS",
	"TWCC_BUMMER (Failure due to unknown causes)",
	"TWCC_LOWMEMORY",
	"TWCC_NODS (No Data Source)",
	"TWCC_MAXCONNECTIONS (DS is connected to max possible apps)",
	"TWCC_OPERATIONERROR (DS or DSM reported error, app shouldn't)",
	"TWCC_BADCAP (Unknown capability)",
	"7 (undefined)",
	"8 (undefined)",
	"TWCC_BADPROTOCOL (Unrecognized triplet)",
	"TWCC_BADVALUE (Data parameter out of range)",
	"TWCC_SEQERROR (Triplet out of sequence)",
	"TWCC_BADDEST (Unknown dest. App/Src in DSM_Entry)",
};

//------------ Global variables

STATIC int			iAvailable;			// TWAIN available: 0:unknown, -1:No, 1:Yes
STATIC int			nState = 1;			// TWAIN state (per the standard)
STATIC int			nErrDetail;			// detailed error code
STATIC unsigned		nErrRC, nErrCC;		// result code and condition code for last error
STATIC char			szMsg[256];			// scratch buffer for messages
STATIC DSMENTRYPROC	pDSM_Entry;			// entry point of Data Source Manager (TWAIN.DLL)
STATIC HANDLE		hDSMLib;			// handle of DSM
STATIC TW_IDENTITY	AppId = {			// application identity structure
	0,									// Id, filled in by DSM
	{ 1, 0, TWLG_USA, TWCY_USA, "<?>"},	// Version
    TWON_PROTOCOLMAJOR,
    TWON_PROTOCOLMINOR,
    DG_IMAGE | DG_CONTROL,
    "<?>",								// Mfg
    "<?>",								// Family
    "<?>"								// Product
    };
STATIC TW_IDENTITY	SourceId;			// source identity structure
STATIC BOOL			bHideUI;			// allow source u/i to be hidden
STATIC TW_USERINTERFACE twUI;
STATIC TW_PENDINGXFERS pendingXfers;
STATIC HANDLE		hDib;				// bitmap returned by native transfer
STATIC TW_INT16		rc;					// result code       
STATIC HINSTANCE	hinstLib;			// instance handle for this DLL

const char szInsuffMem[] = "Insufficient Memory";	// error message

//------------ Forward declarations

void TWAIN_NativeXferReady(LPMSG pmsg);
HWND CreateProxyWindow(void);
unsigned Intersect16(unsigned wMask, unsigned nItems, TW_UINT16 far *pItem);
unsigned BitCount(unsigned W);
void TWAIN_ErrorBox(const char *szMsg);
int RecordError(ErrorDetail ed);
void ClearError(void);
void TWAIN_ReportLastError(LPSTR lpzGeneral);
TW_UINT32 ToFix32(double r);
double Fix32ToFloat(TW_FIX32 fix);


//------------ Public functions


int CALLBACK __export LibMain(HINSTANCE hinst, WORD wDataSeg, WORD cbHeapSize, LPSTR lpszCmdLine)
{
	wDataSeg = wDataSeg; cbHeapSize = cbHeapSize; lpszCmdLine = lpszCmdLine;

	hinstLib = hinst;
	return 1;		// indicate success
} // LibMain


int FAR PASCAL __export _WEP(int x)
{
	x = x;				// suppress 'is never used' warning
	return 1;
} // _WEP


void FAR PASCAL __export TWAIN_RegisterApp(	// record application information
	int		nMajorNum, int nMinorNum,	// major and incremental revision of application. E.g.
										// for version 2.1, nMajorNum == 2 and nMinorNum == 1
	int		nLanguage,					// language of this version (use TWLG_xxx from TWAIN.H)
	int		nCountry,					// country of this version (use TWCY_xxx from TWAIN.H)
	LPSTR	lpszVersion,				// version info string e.g. "1.0b3 Beta release"
	LPSTR	lpszMfg,					// name of manufacturer/developer e.g. "Crazbat Software"
	LPSTR	lpszFamily,					// product family e.g. "BitStomper"
	LPSTR	lpszProduct)				// specific product e.g. "BitStomper Deluxe Pro"
{
    AppId.Id = 0;						// init to 0, but Source Manager will assign real value
    AppId.Version.MajorNum = nMajorNum;
    AppId.Version.MinorNum = nMinorNum;
    AppId.Version.Language = nLanguage;
    AppId.Version.Country  = nCountry;
    lstrcpy (AppId.Version.Info,  lpszVersion);

    AppId.ProtocolMajor =    TWON_PROTOCOLMAJOR;
    AppId.ProtocolMinor =    TWON_PROTOCOLMINOR;
    AppId.SupportedGroups =  DG_IMAGE | DG_CONTROL;
    lstrcpy (AppId.Manufacturer,  lpszMfg);
    lstrcpy (AppId.ProductFamily, lpszFamily);
    lstrcpy (AppId.ProductName,   lpszProduct);

} // TWAIN_RegisterApp


int FAR PASCAL __export TWAIN_SelectImageSource(HWND hwnd)
{
	int fSuccess = FALSE;
	int fProxyWindow = FALSE;
	TW_IDENTITY		NewSourceId;

 	if (!IsWindow(hwnd)) {
 		if (hwnd != NULL) {
 			TWAIN_ErrorBox("TWAIN_SelectImageSource: window handle is invalid");
 			return FALSE;
 		}
 		hwnd = CreateProxyWindow();
 		if (!IsWindow(hwnd)) {
			TWAIN_ErrorBox("TWAIN_SelectImageSource: Unable to create proxy window");
			return FALSE;
 		}
 		fProxyWindow = TRUE;
 	}
	if (TWAIN_LoadSourceManager()) {
		if (TWAIN_OpenSourceManager(hwnd)) {
			// I will settle for the system default.  Shouldn't I get a highlight
			// on system default without this call?
			TWAIN_Mgr(DG_CONTROL, DAT_IDENTITY, MSG_GETDEFAULT, &NewSourceId);
			// now do the real thing
	    	fSuccess = TWAIN_Mgr(DG_CONTROL, DAT_IDENTITY, MSG_USERSELECT, &NewSourceId);
		    TWAIN_CloseSourceManager(hwnd);
	    } else {
			TWAIN_ErrorBox("Unable to open Source Manager (TWAIN.DLL)");
	    }
	    TWAIN_UnloadSourceManager();
	} else {
		TWAIN_ErrorBox("Unable to load Source Manager (TWAIN.DLL)");
	}
    if (fProxyWindow) DestroyWindow(hwnd);

	return fSuccess;
} // TWAIN_SelectImageSource


HANDLE FAR PASCAL __export TWAIN_AcquireNative(HWND hwnd, unsigned wPixTypes)
{
	int fProxyWindow = FALSE;
	hDib = NULL;
	ClearError();			// clear error detail
 
 	if (!IsWindow(hwnd)) {
 		// hwnd isn't a valid window handle - most likely NULL
 		hwnd = CreateProxyWindow();
 		if (hwnd) fProxyWindow = TRUE;
 	}
 	if (!IsWindow(hwnd)) {
		TWAIN_ErrorBox("Unable to create proxy window");
	} else if (!TWAIN_LoadSourceManager()) {
		TWAIN_ErrorBox("Unable to load Source Manager (TWAIN.DLL)");
	} else if (!TWAIN_OpenSourceManager(hwnd)) {
		TWAIN_ErrorBox("Unable to open Source Manager (TWAIN.DLL)");
    } else if (!TWAIN_OpenDefaultSource()) {
		TWAIN_ReportLastError("Unable to open default Data Source.");
	} else if (!TWAIN_NegotiatePixelTypes(wPixTypes)) {
		TWAIN_ReportLastError("Failed to negotiate Pixel Type.");
	} else if (!TWAIN_EnableSource(hwnd)) {
		TWAIN_ReportLastError("Failed to enable Data Source.");
	} else {
		EnableWindow(hwnd, FALSE);
		// source is enabled, wait for transfer or source closed
		TWAIN_ModalEventLoop();

		EnableWindow(hwnd, TRUE);
	}

	// shut everything down in the right sequence
	// these routines do nothing if the corresponding 'open' failed
	TWAIN_DisableSource();
	TWAIN_CloseSource();	
    TWAIN_CloseSourceManager(hwnd);
    TWAIN_UnloadSourceManager();

    if (fProxyWindow) {
    	DestroyWindow(hwnd);
    }

	return hDib;
} // TWAIN_AcquireNative


void FAR PASCAL __export TWAIN_FreeNative(HANDLE hdib)
// Release the memory allocated to a native format image, as returned by TWAIN_AcquireNative.
// (For those coding in C or C++, this is just a call to GlobalFree.)
{
	if (hdib) GlobalFree(hdib);
} // TWAIN_FreeNative



int FAR PASCAL  __export TWAIN_AcquireToClipboard(HWND hwndApp, unsigned wPixTypes)
// Like AcquireNative, but puts the resulting image, if any, into the system clipboard.
// Useful for environments like Visual Basic where it is hard to make direct use of a DIB handle.
// A return value of 1 indicates success, 0 indicates failure.
{
	int fOk = FALSE;
	HANDLE hDib = TWAIN_AcquireNative(hwndApp, wPixTypes);
	if (hDib) {
		if (OpenClipboard(hwndApp)) {
			if (EmptyClipboard()) {
				SetClipboardData(CF_DIB, hDib);
				fOk = TRUE;
				hDib = NULL;
			}
			CloseClipboard();
		}
		if (hDib) {
			// something went wrong, recycle the image
			GlobalFree(hDib);
		}
	}
	return fOk;		// failed
} // TWAIN_AcquireToClipboard


int FAR PASCAL __export TWAIN_IsAvailable(void)
// return 1 if TWAIN services are available, 0 if 'TWAIN-less' system
{
	if (pDSM_Entry) return TRUE;		// SM currently loaded

	if (iAvailable == 0) {
		if (TWAIN_LoadSourceManager()) {
			iAvailable = 1;
			TWAIN_UnloadSourceManager();
		} else {
			iAvailable = -1;
		}
	}
	return (iAvailable > 0);
} // TWAIN_IsAvailable


int FAR PASCAL __export TWAIN_EasyVersion(void)
// Returns the version number of EZTWAIN.DLL, multiplied by 100.
// So e.g. version 2.01 will return 201 from this call.
{
	return VERSION;
} // TWAIN_EasyVersion


int FAR PASCAL __export TWAIN_State(void)
// Returns the TWAIN Protocol State per the spec.
{
	return nState;
} 

int FAR PASCAL __export TWAIN_GetHideUI(void)
{
	return bHideUI;
} // TWAIN_GetHideUI

void FAR PASCAL __export TWAIN_SetHideUI(int fHide)
{
	bHideUI = (fHide != 0);
} // TWAIN_SetHideUI


int FAR PASCAL __export TWAIN_LoadSourceManager(void)
{
	char		szSMDir[128];
	int			cc;
	OFSTRUCT	of;

	if (nState >= 2) return TRUE;			// DSM already loaded

	GetWindowsDirectory(szSMDir, sizeof(szSMDir));
	cc = lstrlen(szSMDir);
	if (cc && szSMDir[cc-1] != ':') {
		lstrcat(szSMDir, "\\");
	}
	lstrcat(szSMDir, "TWAIN.DLL");			// could crash!
	if (OpenFile(szSMDir, &of, OF_EXIST) != -1) {
		hDSMLib = LoadLibrary(szSMDir);
	} else {
		hDSMLib = NULL;
	}
	if (IsValidHandle(hDSMLib)) {
		pDSM_Entry = (DSMENTRYPROC) GetProcAddress(hDSMLib, "DSM_ENTRY");
		if (pDSM_Entry) {
			iAvailable = 1;
			nState = 2;
		} else {
			FreeLibrary(hDSMLib);
			hDSMLib = NULL;
		}
	} else {
		pDSM_Entry = NULL;
	}
	
	return (nState >= 2);
} // TWAIN_LoadSourceManager


int FAR PASCAL __export TWAIN_OpenSourceManager(HWND hwnd)
{
	TW_INT32 hwnd32 = (TW_INT32)hwnd;

	if (TWAIN_LoadSourceManager() &&
		TWAIN_Mgr(DG_CONTROL, DAT_PARENT, MSG_OPENDSM, &hwnd32)) {
		nState = 3;
	}

	return (nState >= 3);
} // TWAIN_OpenSourceManager


int FAR PASCAL __export TWAIN_OpenDefaultSource(void)
{
	if (nState != 3) return FALSE;
	// open the system default source
	SourceId.ProductName[0] = '\0';
	SourceId.Id = 0;
	if (TWAIN_Mgr(DG_CONTROL, DAT_IDENTITY, MSG_OPENDS, &SourceId)) {
		nState = 4;
	} else {
		RecordError(ED_DSM_FAILURE);
	}

	return (nState == 4);
} // TWAIN_OpenDefaultSource


int FAR PASCAL __export TWAIN_EnableSource(HWND hwnd)
{
	if (nState != 4) return FALSE;

	twUI.ShowUI = !bHideUI;
	twUI.hParent = (TW_HANDLE)hwnd;
	TWAIN_DS(DG_CONTROL, DAT_USERINTERFACE, MSG_ENABLEDS, &twUI);
	if (rc == TWRC_FAILURE) {
		RecordError(ED_DS_FAILURE);
	} else {
		// rc could be either SUCCESS or CHECKSTATUS
		nState = 5;
		// note, source will set twUI.ModalUI.
	}
	return (nState == 5);
} // TWAIN_EnableSource


int FAR PASCAL __export TWAIN_DisableSource(void)
{
	if (nState == 5 &&
		TWAIN_DS(DG_CONTROL, DAT_USERINTERFACE, MSG_DISABLEDS, &twUI)) {
		nState = 4;
	}
	return (nState <= 4);
} // TWAIN_DisableSource


int FAR PASCAL __export TWAIN_CloseSource(void)
{
	rc = TWRC_SUCCESS;

	if (nState == 5) TWAIN_DisableSource();
	if (nState == 4 &&
		TWAIN_Mgr(DG_CONTROL, DAT_IDENTITY, MSG_CLOSEDS, &SourceId)) {
		nState = 3;
	}
	return (nState <= 3);
} // TWAIN_CloseSource


int FAR PASCAL __export TWAIN_CloseSourceManager(HWND hwnd)
{
	TW_INT32 hwnd32 = (TW_INT32)hwnd;

	rc = TWRC_SUCCESS;

	if (TWAIN_CloseSource() &&
		TWAIN_Mgr(DG_CONTROL, DAT_PARENT, MSG_CLOSEDSM, &hwnd32)) {
		nState = 2;
	}
	return (nState <= 2);
} // TWAIN_CloseSourceManager


int FAR PASCAL __export TWAIN_UnloadSourceManager(void)
{
	if (nState == 2) {
	
		if (hDSMLib) {
			FreeLibrary(hDSMLib);
			hDSMLib = NULL;
		}
		pDSM_Entry = NULL;
		nState = 1;
	}
	return (nState == 1);
} // TWAIN_UnloadSourceManager



void FAR PASCAL __export TWAIN_ModalEventLoop(void)
{
	MSG msg;

	while ((nState >= 5) && !hDib && GetMessage((LPMSG)&msg, NULL, 0, 0)) {
		if (!TWAIN_MessageHook ((LPMSG)&msg)) {
			TranslateMessage ((LPMSG)&msg);
			DispatchMessage ((LPMSG)&msg);
		}
	} // while
} // TWAIN_ModalEventLoop


int FAR PASCAL __export TWAIN_MessageHook(LPMSG lpmsg)
// returns TRUE if msg processed by TWAIN (source)
{
	int   bProcessed = FALSE;

	if (nState >= 5) {
		// source enabled
		TW_EVENT	twEvent;
		twEvent.pEvent = (TW_MEMREF)lpmsg;
		twEvent.TWMessage = MSG_NULL;
		// see if source wants to process (eat) the message
		TWAIN_DS(DG_CONTROL, DAT_EVENT, MSG_PROCESSEVENT, &twEvent);
		bProcessed = (rc != TWRC_NOTDSEVENT);
		switch (twEvent.TWMessage) {
			case MSG_XFERREADY:
				nState = 6;
				TWAIN_NativeXferReady(lpmsg);
				break;
			case MSG_CLOSEDSREQ:
				TWAIN_DisableSource();
				break;
			case MSG_NULL:
				// no message returned from DS
				break;
		} // switch
    }
	return bProcessed;
} // TWAIN_MessageHook



void TWAIN_NativeXferReady(LPMSG pmsg)
{
	TW_UINT32		hNative;

    pmsg = pmsg;		// suppress warning

	assert(nState == 6);
	TWAIN_DS(DG_IMAGE, DAT_IMAGENATIVEXFER, MSG_GET, &hNative);

	if (rc != TWRC_XFERDONE) hDib = NULL;

	switch (rc) {
		case TWRC_XFERDONE:
			// hNative contains a valid native image (handle)
			// application is responsible for de-allocating.
			nState = 7;
			// Need to acknowledge the end of the transfer
			hDib = (HANDLE)hNative;
			break;

		case TWRC_CANCEL:
			// user cancelled the transfer
			// hNative is invalid
			nState = 7;
			// acknowledge the end of the transfer
			break;

		case TWRC_FAILURE:
		default:
			// the transfer failed (e.g. insufficient memory)
			// hNative is invalid
			// check condition code for more info
			nState = 6;
			// state transition failed
			// image data is still pending
			break;
	} // switch
	assert(nState >= 6);

	TWAIN_AbortAllPendingXfers();	// state 7 or 6 -> state 5

} // TWAIN_NativeXferReady


int FAR PASCAL __export TWAIN_AbortAllPendingXfers(void)
{
	if (nState == 7 &&
		TWAIN_DS(DG_CONTROL, DAT_PENDINGXFERS, MSG_ENDXFER, &pendingXfers)) {
		nState = pendingXfers.Count ? 6 : 5;
	}
	if (nState == 6 &&
		TWAIN_DS(DG_CONTROL, DAT_PENDINGXFERS, MSG_RESET, &pendingXfers)) {
		nState = 5;
	}
	return (nState <= 5);
} // TWAIN_AbortAllPendingXfers



int FAR PASCAL __export TWAIN_WriteNativeToFilename(HANDLE hdib, LPCSTR pszFile)
// Writes a DIB handle to a .BMP file
//
// hdib		= DIB handle, as returned by TWAIN_AcquireNative
// pszFile	= far pointer to NUL-terminated filename
// If pszFile is NULL or points to a null string, prompts the user
// for the filename with a standard file-save dialog.
//
// Return values:
//	 0	success
//	-1	user cancelled File Save dialog
//	-2	file open error (invalid path or name, or access denied)
//	-3	(weird) unable to lock DIB - probably an invalid handle.
//	-4	writing BMP data failed, possibly output device is full
{
	int result;
	char szFile[256];
	HFILE fh;
	OFSTRUCT ofs;

	if (!pszFile || !*pszFile) {
		// prompt for filename
		OPENFILENAME ofn;
		int nExt;
	
		_fmemset(&ofn, 0, sizeof ofn);
		szFile[0] = '\0';
		ofn.lStructSize = sizeof(OPENFILENAME);
		ofn.hwndOwner = NULL;
		ofn.lpstrFilter = "Windows Bitmap (*.bmp)\0*.bmp\0\0";
		ofn.lpstrFile= szFile;
		ofn.nMaxFile = sizeof(szFile) - 5;
		ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY |
					OFN_NOREADONLYRETURN;
	
		if (!GetSaveFileName(&ofn)) {
			return -1;					// user cancelled dialog
		}
		// supply default extension - GetSaveFileName doesn't seem to do it!
		nExt = ofn.nFileExtension;
		if (nExt && !szFile[nExt]) {
			// no extension
			lstrcat(szFile, ".bmp");
		}
		pszFile = szFile;
	}

	result = -2;
	fh = OpenFile(pszFile , &ofs, OF_CREATE | OF_WRITE);
	if (fh != HFILE_ERROR) {
		result = TWAIN_WriteNativeToFile(hdib, fh);
		_lclose(fh);
	}
	return result;
} // TWAIN_WriteNativeToFilename


int FAR PASCAL __export TWAIN_WriteNativeToFile(HANDLE hdib, HFILE fh)
// Writes a DIB to a file in .BMP format.
//
// hdib		= DIB handle, as returned by TWAIN_AcquireNative
// fh		= file handle, as returned by C _open or Windows _lopen or OpenFile
//
// Return value as for TWAIN_WriteNativeToFilename
{
	int result = -3;
	LPBITMAPINFOHEADER lpbmi = (LPBITMAPINFOHEADER)GlobalLock(hdib);
	if (lpbmi) {
		result = -4;
		if (TWAIN_WriteDibToFile(lpbmi, fh)) {
			result = 0;			// success
		}
		GlobalUnlock(hdib);
	}
	return result;
} //  TWAIN_WriteNativeToFile


int FAR PASCAL __export TWAIN_WriteDibToFile(LPBITMAPINFOHEADER lpDIB, HFILE fh)
{
	BITMAPFILEHEADER		bfh;
	int						fOk = FALSE;
	int						nBPP = lpDIB->biBitCount;
	int						nColors = (int)lpDIB->biClrUsed;

	// figure out actual size of color table
	if (nColors == 0 && nBPP <= 8) {
		nColors = (1 << nBPP);
	}
	if (lpDIB->biCompression == BI_RGB) {
		// uncompressed bitmap, image size might not be set
		DWORD dwBytesPerRow = (((lpDIB->biWidth * nBPP) + 31) / 32) * 4;
		lpDIB->biSizeImage = dwBytesPerRow * lpDIB->biHeight;
	} else if (lpDIB->biSizeImage == 0) {
		// compressed bitmap, image size had damn well better be set!
		return FALSE;
		// This could be hacked around with something like this:
		//lpDIB->biSizeImage = GlobalSize((HANDLE)GlobalHandle(HIWORD(lpDIB)));
	}

	// Set up BMP header.
	bfh.bfType = 0x4d42;                // "BM"
	bfh.bfReserved1 = 0;
	bfh.bfReserved2 = 0;
	bfh.bfOffBits = sizeof(BITMAPFILEHEADER) +
					sizeof(BITMAPINFOHEADER) +
					sizeof(RGBQUAD) * nColors;
	bfh.bfSize = bfh.bfOffBits + lpDIB->biSizeImage;

	if (_lwrite(fh, (LPSTR)&bfh, sizeof bfh) == sizeof bfh) {

		long towrite = bfh.bfSize - (long)sizeof bfh;

		if (_hwrite(fh, lpDIB, towrite) == towrite) {
			fOk = TRUE;
		}
	}

	return fOk;

} // TWAIN_WriteDibToFile


int FAR PASCAL __export TWAIN_NegotiateXferCount(int nXfers)
{
	return TWAIN_SetCapOneValue(CAP_XFERCOUNT, TWTY_INT16, nXfers);
} // TWAIN_NegotiateXferCount


int FAR PASCAL __export TWAIN_NegotiatePixelTypes(unsigned wPixTypes)
{
	TW_CAPABILITY 		cap;
	void far *			pv;
	int					fSuccess = FALSE;

	if (nState != TWAIN_SOURCE_OPEN) {
		return RecordError(ED_NOT_STATE_4);
	}

	if (TWAIN_ANYTYPE == wPixTypes) {
		return TRUE;			// that was easy!
	}
                                                                           
                                                                           
	// Fill in capability structure
	cap.Cap = ICAP_PIXELTYPE;			// capability id
	cap.ConType = TWON_ENUMERATION;		// favorite type of container (should be ignored...)

	if (!TWAIN_DS(DG_CONTROL, DAT_CAPABILITY, MSG_GET, (TW_MEMREF)&cap)) {
		return RecordError(ED_CAP_GET);
	}
	if (!cap.hContainer) {
		return RecordError(ED_NULL_HCON);
	}
	if (!(pv = GlobalLock(cap.hContainer))) {
		// this source is invalid, further negotiation is unlikely to succeed
		return RecordError(ED_BAD_HCON);
	}

	switch (cap.ConType) {

		case TWON_ENUMERATION: {
			TW_ENUMERATION far *pcon = (TW_ENUMERATION far *)pv;
			if (pcon->NumItems < 1) {
				RecordError(ED_CAP_GET_EMPTY);
			} else if (pcon->ItemType != TWTY_UINT16 && pcon->ItemType != TWTY_INT16) {
				RecordError(ED_BAD_ITEMTYPE);
			} else {
				pcon->CurrentIndex = TWON_DONTCARE32;	// don't change current value
				pcon->DefaultIndex = TWON_DONTCARE32;	// don't change default value
				pcon->NumItems = Intersect16(wPixTypes, (unsigned)pcon->NumItems, (TW_UINT16 far *)pcon->ItemList);
				fSuccess = (pcon->NumItems != 0);
				if (!fSuccess) RecordError(ED_CAP_SET_EMPTY);
			}
			break;
		}
		
		case TWON_ARRAY: {
			// this is technically illegal - TWAIN 1.5, p9-30, Containers for MSG_GET: TW_ENUMERATION, TW_ONEVALUE
			TW_ARRAY far *pcon = (TW_ARRAY far *)pv;
			if (pcon->NumItems < 1) {
				RecordError(ED_CAP_GET_EMPTY);
			} else if (pcon->ItemType != TWTY_UINT16 && pcon->ItemType != TWTY_INT16) {
				RecordError(ED_BAD_ITEMTYPE);
			} else {
				pcon->NumItems = Intersect16(wPixTypes, (unsigned)pcon->NumItems, (TW_UINT16 far *)pcon->ItemList);
				fSuccess = (pcon->NumItems != 0);
				if (!fSuccess) RecordError(ED_CAP_SET_EMPTY);
			}
			break;
		}
		
		case TWON_ONEVALUE: {
			TW_ONEVALUE far *pcon = (TW_ONEVALUE far *)pv;
			fSuccess = ((1 << pcon->Item) & wPixTypes);
			if (!fSuccess) RecordError(ED_CAP_SET_EMPTY);
			break;
		}
		
		default:
			// something we don't understand, abandon negotiations
			RecordError(ED_BAD_CONTYPE);
			break;
	} // switch

	GlobalUnlock(cap.hContainer);

	if (fSuccess) {
		// For enums (and arrays) send intersection back, to restrict it.
		// For one vals, don't bother - could only cause confusion.
		if (cap.ConType != TWON_ONEVALUE) {
			fSuccess = TWAIN_DS(DG_CONTROL, DAT_CAPABILITY, MSG_SET, (TW_MEMREF)&cap);
			if (!fSuccess) RecordError(ED_CAP_SET);
		}
	}
	GlobalFree(cap.hContainer);
	
	return fSuccess;
} // TWAIN_NegotiatePixelTypes


int FAR PASCAL __export TWAIN_GetCurrentUnits(void)
{
	int nUnits = TWUN_INCHES;
	TWAIN_GetCapCurrent(ICAP_UNITS, TWTY_UINT16, &nUnits);
	return nUnits;
} // TWAIN_GetCurrentUnits


int FAR PASCAL __export TWAIN_SetCurrentUnits(int nUnits)
// Negotiate the current pixel type for acquisition.
// Negotiation is only allowed in State 4 (TWAIN_SOURCE_OPEN)
// The source may select this pixel type, but don't assume it will.
{
	return TWAIN_SetCapOneValue(ICAP_UNITS, TWTY_UINT16, (TW_UINT16)nUnits);
} // TWAIN_SetCurrentUnits


int FAR PASCAL __export TWAIN_GetBitDepth(void)
// Ask the source for the current bitdepth.
// This value depends on the current PixelType.
// Bit depth is per color channel e.g. 24-bit RGB has bit depth 8.
// If anything goes wrong, this function returns 0.
{
	int nBits = 0;
	TWAIN_GetCapCurrent(ICAP_BITDEPTH, TWTY_UINT16, &nBits);
	return nBits;
} // TWAIN_GetBitDepth

int FAR PASCAL __export TWAIN_SetBitDepth(int nBits)
// (Try to) set the current bitdepth (for the current pixel type).
{
	return TWAIN_SetCapOneValue(ICAP_BITDEPTH, TWTY_UINT16, (TW_UINT16)nBits);
} // TWAIN_SetBitDepth


int FAR PASCAL __export TWAIN_GetPixelType(void)
// Ask the source for the current pixel type.
// If anything goes wrong (it shouldn't!), this function returns 0 (TWPT_BW).
{
	int nPixType = 0;
	TWAIN_GetCapCurrent(ICAP_PIXELTYPE, TWTY_UINT16, &nPixType);
	return nPixType;
} // TWAIN_GetPixelType

int FAR PASCAL __export TWAIN_SetCurrentPixelType(int nPixType)
// Negotiate the current pixel type for acquisition.
// Negotiation is only allowed in State 4 (TWAIN_SOURCE_OPEN)
// The source may select this pixel type, but don't assume it will.
{
	return TWAIN_SetCapOneValue(ICAP_PIXELTYPE, TWTY_UINT16, (TW_UINT16)nPixType);
} // TWAIN_SetCurrentPixelType


double FAR PASCAL __export TWAIN_GetCurrentResolution(void)
{
	TW_FIX32 res;
	TWAIN_GetCapCurrent(ICAP_XRESOLUTION, TWTY_FIX32, &res);
	
	return Fix32ToFloat(res);
} // TWAIN_GetCurrentResolution

int FAR PASCAL __export TWAIN_SetCurrentResolution(double dRes)
// Negotiate the current resolution for acquisition.
// Negotiation is only allowed in State 4 (TWAIN_SOURCE_OPEN)
// The source may select this resolution, but don't assume it will.
{
	return TWAIN_SetCapOneValue(ICAP_XRESOLUTION, TWTY_FIX32, ToFix32(dRes));
} // TWAIN_SetCurrentResolution


int FAR PASCAL __export TWAIN_SetCapOneValue(unsigned Cap, unsigned ItemType, long ItemVal)
{
	TW_CAPABILITY	cap;
	pTW_ONEVALUE	pv;
	BOOL			bSuccess;

	if (nState != TWAIN_SOURCE_OPEN) {
		TWAIN_ErrorBox("Attempt to set capability outside State 4.");
		return FALSE;
	}

	cap.Cap = Cap;			// capability id
	cap.ConType = TWON_ONEVALUE;		// container type
	cap.hContainer = GlobalAlloc(GHND, sizeof (TW_ONEVALUE));
	if (!cap.hContainer) {
		TWAIN_ErrorBox(szInsuffMem);
		return FALSE;
	}
	pv = (pTW_ONEVALUE)GlobalLock(cap.hContainer);
	pv->ItemType = ItemType;
	pv->Item = ItemVal;
	GlobalUnlock(cap.hContainer);
	bSuccess = TWAIN_DS(DG_CONTROL, DAT_CAPABILITY, MSG_SET, (TW_MEMREF)&cap);
	GlobalFree(cap.hContainer);
	return bSuccess;
} // TWAIN_SetCapOneValue


const size_t nTypeSize[13] =
	{	sizeof (TW_INT8),
		sizeof (TW_INT16),
		sizeof (TW_INT32),
		sizeof (TW_UINT8),
		sizeof (TW_UINT16),
		sizeof (TW_UINT32),
		sizeof (TW_BOOL),
		sizeof (TW_FIX32),
		sizeof (TW_FRAME),
		sizeof (TW_STR32),
		sizeof (TW_STR64),
		sizeof (TW_STR128),
		sizeof (TW_STR255),
	};

// helper function:
int TypeMatch(unsigned nTypeA, unsigned nTypeB)
{
	// Integral types match if they are the same size.
	// All other types match only if they are equal
	return nTypeA == nTypeB ||
		   (nTypeA <= TWTY_UINT32 &&
		    nTypeB <= TWTY_UINT32 &&
		    nTypeSize[nTypeA] == nTypeSize[nTypeB]);
} // TypeMatch



int FAR PASCAL __export TWAIN_GetCapCurrent(unsigned Cap, unsigned ItemType, void FAR *pVal)
{
	TW_CAPABILITY 	cap;
	void far *		pv = NULL;
	BOOL			bSuccess = FALSE;

	assert(pVal != NULL);

	if (nState < TWAIN_SOURCE_OPEN) {
		TWAIN_ErrorBox("Attempt to get capability value below State 4.");
		return FALSE;
	}

	// Fill in capability structure
	cap.Cap = Cap;					// capability id
	cap.ConType = TWON_ONEVALUE;	// favorite type of container (should be ignored...)
	cap.hContainer = NULL;

	if (TWAIN_DS(DG_CONTROL, DAT_CAPABILITY, MSG_GETCURRENT, (TW_MEMREF)&cap) &&
	    cap.hContainer &&
	    (pv = GlobalLock(cap.hContainer))) {

		if (cap.ConType == TWON_ENUMERATION) {
			TW_ENUMERATION far *pcon = (TW_ENUMERATION far *)pv;
			TW_UINT32 index = pcon->CurrentIndex;
			if (index < pcon->NumItems && TypeMatch(pcon->ItemType, ItemType)) {
				LPSTR pitem = (LPSTR)pcon->ItemList + index*nTypeSize[ItemType];
				_fmemcpy(pVal, pitem, nTypeSize[ItemType]);
				bSuccess = TRUE;
			}
		} else if (cap.ConType == TWON_ONEVALUE) {
			TW_ONEVALUE far *pcon = (TW_ONEVALUE far *)pv;
			if (TypeMatch(pcon->ItemType, ItemType)) {
				_fmemcpy(pVal, &pcon->Item, nTypeSize[ItemType]);
				bSuccess = TRUE;
			}
		}
	}
	
	if (pv) GlobalUnlock(cap.hContainer);
	if (cap.hContainer) GlobalFree(cap.hContainer);

	return bSuccess;
} 

//-------------------------- The primitive functions


int FAR PASCAL __export TWAIN_DS(unsigned long dg, unsigned dat, unsigned msg, void FAR *pd)
// Call the current source with a triplet
{
	assert(nState >= 4);
	rc = TWRC_FAILURE;
	if (pDSM_Entry) {
		rc = (*pDSM_Entry)(&AppId, &SourceId, dg,dat,msg, (TW_MEMREF)pd);
	}
	return (rc == TWRC_SUCCESS);
} // TWAIN_DS



int FAR PASCAL __export TWAIN_Mgr(unsigned long dg, unsigned dat, unsigned msg, void FAR *pd)
// Call the Source Manager with a triplet
{
	rc = TWRC_FAILURE;
	if (pDSM_Entry) {
		rc = (*pDSM_Entry)(&AppId, NULL, dg, dat, msg, (TW_MEMREF)pd);
	}
	return (rc == TWRC_SUCCESS);
} // TWAIN_Mgr



unsigned FAR PASCAL __export TWAIN_GetResultCode(void)
{
	return rc;
} // TWAIN_GetResultCode



unsigned FAR PASCAL __export TWAIN_GetConditionCode(void)
{
	TW_STATUS	twStatus;

	if (nState >= 4) {
		// get source status if open
		TWAIN_DS(DG_CONTROL, DAT_STATUS, MSG_GET, (TW_MEMREF)&twStatus);
	} else if (nState == 3) {
		// otherwise get source manager status
		TWAIN_Mgr(DG_CONTROL, DAT_STATUS, MSG_GET, (TW_MEMREF)&twStatus);
	} else {
		// nothing open, not a good time to get condition code!
		return TWCC_SEQERROR;
	}
	if (rc == TWRC_SUCCESS) {
		return twStatus.ConditionCode;
	} else {
		return TWCC_BUMMER;			// what can I say. 
	}
} // TWAIN_GetConditionCode


//------------ Private functions


HWND CreateProxyWindow(void)
{
	HWND hwnd;
	hwnd = CreateWindow("STATIC",						// class
						"Acquire Proxy",				// title
						WS_POPUPWINDOW | WS_VISIBLE,	// style
						CW_USEDEFAULT, CW_USEDEFAULT,	// x, y
						CW_USEDEFAULT, CW_USEDEFAULT,	// width, height
						HWND_DESKTOP,					// parent window
						NULL,							// hmenu
						hinstLib,						// hinst
						NULL);							// lpvparam
	return hwnd;
} // CreateProxyWindow


int RecordError(ErrorDetail ed)
{
	if (nErrDetail == ED_NONE) {
		nErrDetail = ed;
		if (ed > ED_START_TRIPLET_ERRS && ed < ED_END_TRIPLET_ERRS) {
			nErrRC = TWAIN_GetResultCode();
			nErrCC = TWAIN_GetConditionCode();
		} else {
			nErrRC = 0;
			nErrCC = 0;
		}
	}
	return FALSE;
} // RecordError


void ClearError(void)
{
	nErrDetail = ED_NONE;
} // ClearError


void TWAIN_ReportLastError(LPSTR lpzGeneral)
{
	if (nErrDetail > ED_START_TRIPLET_ERRS && nErrDetail < ED_END_TRIPLET_ERRS) {
		wsprintf(szMsg, "%s\n%s\nRC:   %s\nCC:   %s",
			lpzGeneral, (LPSTR)pszErrDescrip[nErrDetail],
			(LPSTR)pszRC[nErrRC], (LPSTR)pszCC[nErrCC]);
	} else {
		wsprintf(szMsg, "%s\n%s", lpzGeneral, (LPSTR)pszErrDescrip[nErrDetail]);
	}
	TWAIN_ErrorBox(szMsg);
} // ReportLastError


void TWAIN_ErrorBox(const char *szMsg)
{
	MessageBox(NULL, (LPSTR)szMsg, "TWAIN Error", MB_ICONEXCLAMATION | MB_OK);
} // TWAIN_ErrorBox


unsigned BitCount(unsigned W)
{
	unsigned n = 0;

	while (W) {
		n += (W & 1);
		W >>= 1;
	} // while
	return n;
} // BitCount


unsigned Intersect16(unsigned wMask, unsigned nItems, TW_UINT16 far *pItem)
{
	unsigned	wSet;
	unsigned	i;

	// In wSet, construct set of available items.
	// Note that items that cannot be represented in wMask are also
	// unrepresentable in wSet so are implicitly discarded
	for (i = wSet = 0 ; i < nItems; i++) {
		wSet |= 1 << pItem[i];
	} // for

	// Discard anything in wMask that isn't in wSet
	wMask &= wSet;

	// Re-fill the item table with intersection set
	for (i = nItems = 0; wMask ; wMask>>=1,i++) {
		if (wMask & 1) {
			pItem[nItems++] = i;
		}
	} // for

	return nItems;
}


TW_UINT32 ToFix32(double r)
{
	TW_FIX32 fix;
	TW_INT32 val = (TW_INT32)(r * 65536.0 + 0.5);
	fix.Whole = (TW_INT16)(val >> 16);			// most significant 16 bits
	fix.Frac = (TW_UINT16)(val & 0xffff);		// least
	assert(sizeof (TW_FIX32) == sizeof (TW_UINT32));
	return *(TW_UINT32*)&fix;
} // ToFix32

double Fix32ToFloat(TW_FIX32 fix)
{
	TW_INT32 val = ((TW_INT32)fix.Whole << 16) | ((TW_UINT32)fix.Frac & 0xffff);
	return val / 65536.0;
} // Fix32ToFloat

//--------- END ---------
