{-----------------------------------------------------------------------------}
{   DateTime.pas                                                              }
{                                                                             }
{   The Fred Nurke date/time editing controls for Delphi.                     }
{                                                                             }
{   Copyright 1999 by D. H. Rifkind <dave@subnormal.com>                      }
{   A product of Subnormal Intelligence.                                      }
{     http://www.subnormal.com                                                }
{-----------------------------------------------------------------------------}

unit DateTime;

interface

{ Define RXLIB to use the RX Library spin button. }
{$DEFINE RXLIB}

{ BTNKIND defined if both buttons available. }
{$IFDEF RXLIB}
 {$IFDEF WIN32}
  {$DEFINE BTNKIND}
 {$ENDIF}
{$ENDIF}

{ SPIN defined if either button available. }
{$IFDEF RXLIB}
 {$DEFINE SPIN}
{$ENDIF}
{$IFDEF WIN32}
 {$DEFINE SPIN}
{$ENDIF}

uses Classes, {$IFDEF WIN32}ComCtrls, CommCtrl,{$ENDIF} Consts, Controls, DB,
  DBCtrls, DBTables, Forms, Graphics, Mask, Menus, Messages,
  {$IFDEF RXLIB}RxSpin,{$ENDIF} StdCtrls, SysUtils,
  {$IFDEF WIN32}Windows{$ELSE}WinTypes, WinProcs{$ENDIF};

type
{$IFDEF BTNKIND}
  TSpinButtonKind = RxSpin.TSpinButtonKind;
{$ENDIF}

  TEpoch = 0..9999;

  TCustomDateTimeEdit = class(TCustomMaskEdit)
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Validate;
  protected
    FChanging: Boolean;
    procedure Change; override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure UpdateFormat(WeekDay: Word);
    procedure UpDownButtonClick(Up: Boolean); virtual;
  private
    FArrowKeys: Boolean;
    FBlankChar: Char;
    FDateFormat: String;
    FEditorEnabled: Boolean;
    FEpoch: TEpoch;
    FFourDigitYear: Boolean;
    FSpinButton: Boolean;
    FLongestAMPMString: Integer;
    FLongestDayName: Integer;
    FLongestMonthName: Integer;
    FMask: String;
    FTemplate: String;
    FEvaluated: Boolean;
    FDateTime: TDateTime;
    FBlank: Boolean;
    FValid: Boolean;
{$IFDEF RXLIB}
    FButton: TRxSpinButton;
    FBtnWindow: TWinControl;
{$ENDIF}
{$IFDEF WIN32}
    FUpDown: TCustomUpDown;
{$ENDIF}
{$IFDEF BTNKIND}
    FButtonKind: TSpinButtonKind;
    function GetButtonKind: TSpinButtonKind;
    procedure SetButtonKind(Value: TSpinButtonKind);
{$ENDIF}
{$IFDEF WIN32}
    procedure UpDownClick(Sender: TObject; Button: TUDBtnType);
{$ENDIF}
    function GetButtonWidth: Integer;
    procedure RecreateButton;
    procedure ResizeButton;
    procedure SetEditRect;
    procedure UpClick(Sender: TObject);
    procedure DownClick(Sender: TObject);
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure WMCut(var Message: TWMCut); message WM_CUT;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure WMPaste(var Message: TWMPaste); message WM_PASTE;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure Evaluate;
    function FormatHasWeekDay: Boolean;
    function GetBlank: Boolean;
    function GetDateTime: TDateTime;
    function GetValid: Boolean;
    procedure SetBlank(Value: Boolean);
    procedure SetBlankChar(Value: Char);
    procedure SetDateFormat(const Value: String);
    procedure SetDateTime(Value: TDateTime);
    procedure SetFourDigitYear(Value: Boolean);
    procedure SetSpinButton(Value: Boolean);
  protected
    property ArrowKeys: Boolean read FArrowKeys write FArrowKeys default true;
    property BlankChar: Char read FBlankChar write SetBlankChar default ' ';
{$IFDEF BTNKIND}
    property ButtonKind: TSpinButtonKind read GetButtonKind write SetButtonKind
      default bkDiagonal;
{$ENDIF}
    property DateFormat: String read FDateFormat write SetDateFormat;
    property DateTime: TDateTime read GetDateTime write SetDateTime;
    { Define after DateTime: }
    property Blank: Boolean read GetBlank write SetBlank;
    property EditorEnabled: Boolean read FEditorEnabled write FEditorEnabled
      default true;
    property Epoch: TEpoch read FEpoch write FEpoch default 1900;
    property FourDigitYear: Boolean read FFourDigitYear write SetFourDigitYear
      default false;
    property SpinButton: Boolean read FSpinButton write SetSpinButton default true;
    property Valid: Boolean read GetValid;
  end;

  TDateTimeEdit = class(TCustomDateTimeEdit)
  published
    property ArrowKeys;
    property AutoSelect;
    property AutoSize;
    property BlankChar;
    property BorderStyle;
{$IFDEF BTNKIND}
    property ButtonKind;
{$ENDIF}
    property CharCase;
    property Color;
    property Ctl3D;
    property DateFormat;
    property DateTime;
    { Define after DateTime: }
    property Blank;
    property DragCursor;
    property DragMode;
    property EditorEnabled;
    property Enabled;
    property Epoch;
    property Font;
    property FourDigitYear;
{$IFDEF WIN32}
 {$IFDEF VER90}
    property ImeMode;
    property ImeName;
 {$ENDIF}
{$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property SpinButton;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
{$IFDEF WIN32}
    property OnStartDrag;
{$ENDIF}
  public
    property EditText;
    property Text;
    property Valid;
  end;

  TDBDateTimeEdit = class(TCustomDateTimeEdit)
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  protected
    procedure Change; override;
    function EditCanModify: Boolean; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure Reset; override;
    procedure UpDownButtonClick(Up: Boolean); override;
  private
    FDataLink: TFieldDataLink;
    procedure DataChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    procedure UpdateData(Sender: TObject);
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
{$IFDEF WIN32}
    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
{$ENDIF}
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    function GetDataField: String;
    function GetDataSource: TDataSource;
    function GetField: TField;
    function GetReadOnly: Boolean;
    procedure SetDataField(const Value: String);
    procedure SetDataSource(Value: TDataSource);
    procedure SetReadOnly(Value: Boolean);
  published
    property ArrowKeys;
    property AutoSelect;
    property AutoSize;
    property BlankChar;
    property BorderStyle;
{$IFDEF BTNKIND}
    property ButtonKind;
{$ENDIF}
    property CharCase;
    property Color;
    property Ctl3D;
    property DataField: String read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DateFormat;
    property DragCursor;
    property DragMode;
    property EditorEnabled;
    property Enabled;
    property Epoch;
    property Font;
    property FourDigitYear;
{$IFDEF WIN32}
 {$IFDEF VER90}
    property ImeMode;
    property ImeName;
 {$ENDIF}
{$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default false;
    property ShowHint;
    property SpinButton;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
{$IFDEF WIN32}
    property OnStartDrag;
{$ENDIF}
  public
    property Blank;
    property DateTime;
    property EditText;
    property Field: TField read GetField;
    property Text;
    property Valid;
  end;

implementation

function Pad(const S: String; Width: Integer): String;
begin
  Result := Format('%-*s', [Width, S]);
end;

{$IFNDEF WIN32}
function StringOfChar(C: Char; Count: Integer): String;
begin
  Result := '';
  while Length(Result) < Count do
    AppendStr(Result, C);
end;

function Trim(const S: String): String;
var
  Len: Integer;
  I: Integer;
  J: Integer;
begin
  Len := Length(S);
  I := 1;
  while I <= Len do begin
    if not (S[I] in [#0..#31, ' ']) then break;
    Inc(I);
  end;
  J := Len;
  while J >= I do begin
    if not (S[J] in [#0..#31, ' ']) then break;
    Dec(J);
  end;
  Result := Copy(S, I, J - I + 1);
end;
{$ENDIF}

{$IFDEF WIN32}
type
  { Stolen, pretty much as is, from RxSpin.pas. }
  TDateTimeUpDown = class(TCustomUpDown)
  public
    constructor Create(AOwner: TComponent); override;
  private
    FChanging: Boolean;
    procedure ScrollMessage(var Message: TWMVScroll);
    procedure WMHScroll(var Message: TWMHScroll); message CN_HSCROLL;
    procedure WMVScroll(var Message: TWMVScroll); message CN_VSCROLL;
  protected
    property OnClick;
  end;

constructor TDateTimeUpDown.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Orientation := udVertical;
  Min := -1;
  Max := 1;
  Position := 0;
end;

procedure TDateTimeUpDown.ScrollMessage(var Message: TWMVScroll);
begin
  if Message.ScrollCode = SB_THUMBPOSITION then begin
    if not FChanging then begin
      FChanging := true;
      try
        if Message.Pos > 0 then Click(btNext)
        else if Message.Pos < 0 then Click(btPrev);
        if HandleAllocated then
          SendMessage(Handle, UDM_SETPOS, 0, 0);
      finally
        FChanging := false;
      end;
    end;
  end;
end;

procedure TDateTimeUpDown.WMHScroll(var Message: TWMHScroll);
begin
  ScrollMessage(TWMVScroll(Message));
end;

procedure TDateTimeUpDown.WMVScroll(var Message: TWMVScroll);
begin
  ScrollMessage(Message);
end;
{$ENDIF}


{ TCustomDateTimeEdit }

constructor TCustomDateTimeEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FArrowKeys := true;
  FBlankChar := ' ';
{$IFDEF BTNKIND}
  FButtonKind := bkDiagonal;
{$ENDIF}
  FEditorEnabled := true;
  FEpoch := 1900;
{$IFDEF SPIN}
  FSpinButton := true;
{$ENDIF}
  RecreateButton;
  UpdateFormat(0);
  Modified := false;
end;

destructor TCustomDateTimeEdit.Destroy;
begin
{$IFDEF RXLIB}
  FButton := nil;
{$ENDIF}
{$IFDEF WIN32}
  FUpDown := nil;
{$ENDIF}
  inherited Destroy;
end;

procedure TCustomDateTimeEdit.Validate;
var
  Msg: {$IFDEF WIN32}String{$ELSE}Word{$ENDIF};
  Template: String;
  DateChars: Boolean;
  TimeChars: Boolean;
  Contents: String;
begin
  if not Valid then begin
    Template := UpperCase(FTemplate);
    DateChars := (Pos('Y', Template) > 0) or (Pos('M', Template) > 0) or
                 (Pos('D', Template) > 0);
    TimeChars := (Pos('H', Template) > 0) or (Pos('R', Template) > 0) or
                 (Pos('N', Template) > 0) or (Pos('S', Template) > 0) or
                 (Pos('A', Template) > 0) or (Pos('B', Template) > 0) or
                 (Pos('W', Template) > 0);
    if      DateChars and not TimeChars then Msg := SInvalidDate
    else if TimeChars and not DateChars then Msg := SInvalidTime
    else                                     Msg := SInvalidDateTime;
    Contents := Trim(FormatMaskText(FMask + ';0; ', Text));
{$IFDEF WIN32}
    raise EConvertError.CreateFmt(Msg, [Contents]);
{$ELSE}
    raise EConvertError.CreateResFmt(Msg, [Contents]);
{$ENDIF}
  end;
end;

procedure TCustomDateTimeEdit.Change;
begin
  if not FChanging then begin
    inherited Change;
    FEvaluated := false;
    if (Trim(Text) = '') and FormatHasWeekDay then
      UpdateFormat(0);
  end;
end;

procedure TCustomDateTimeEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  { EM_SETRECTNP only works for multi-line controls.  See also WMGetDlgCode. }
  Params.Style := Params.Style or ES_MULTILINE or WS_CLIPCHILDREN;
end;

procedure TCustomDateTimeEdit.CreateWnd;
begin
  inherited CreateWnd;
  SetEditRect;
end;

procedure TCustomDateTimeEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if FArrowKeys and (Key in [VK_UP, VK_DOWN]) then begin
    if Key = VK_UP then UpClick(Self)
    else if Key = VK_DOWN then DownClick(Self);
    Key := 0;
  end;
  if not FEditorEnabled then begin
    if Key in [VK_BACK, VK_DELETE] then begin
      MessageBeep(0);
      Key := 0;
    end;
  end;
  { N.B. TCustomMaskEdit eats arrow keys. }
  inherited KeyDown(Key, Shift);
end;

procedure TCustomDateTimeEdit.KeyPress(var Key: Char);
begin
  if not FEditorEnabled then begin
    if Key in [#32..#255] then begin
      MessageBeep(0);
      Key := #0;
    end;
  end;
  inherited KeyPress(Key);
end;

function DaysInMonth(Year, Month: Word): Word;
begin
  Inc(Month);
  if Month > 12 then begin
    Inc(Year);
    Month := 1;
  end;
  try
    DecodeDate(EncodeDate(Year, Month, 1) - 1, Year, Month, Result);
  except
    on EConvertError do
      Result := 0;
  end;
end;

procedure TCustomDateTimeEdit.UpDownButtonClick(Up: Boolean);
var
  Year:   Word;
  Month:  Word;
  Day:    Word;
  Hour:   Word;
  Minute: Word;
  Second: Word;
  Millis: Word;
  Fld: Integer;
  Ext: Integer;
  Off: Integer;
  Len: Integer;
begin
  if ReadOnly or not Valid or Blank then MessageBeep(0)
  else begin
    DecodeDate(DateTime, Year, Month, Day);
    DecodeTime(DateTime, Hour, Minute, Second, Millis);
    Len := Length(FTemplate);
    Fld := SelStart + SelLength;
    while (Fld > 1) and (FTemplate[Fld - 1] = FTemplate[Fld]) do
      Dec(Fld);
    Ext := 1;
    while (Fld + Ext <= Len) and (FTemplate[Fld] = FTemplate[Fld + Ext]) do
      Inc(Ext);
    Off := Fld + Ext - SelStart - SelLength;
    case UpCase(FTemplate[Fld]) of
      'Y':  begin
              if Up then Inc(Year) else Dec(Year);
              if Day > DaysInMonth(Year, Month) then
                Day := DaysInMonth(Year, Month);
            end;
      'M':  begin
              if Up then begin
                if Month < 12 then Inc(Month) else Month := 1;
              end else begin
                if Month > 1 then Dec(Month) else Month := 12;
              end;
              if Day > DaysInMonth(Year, Month) then
                Day := DaysInMonth(Year, Month);
            end;
      'D':  begin
              if Up then begin
                if Day < DaysInMonth(Year, Month) then Inc(Day)
                else Day := 1;
              end else begin
                if Day > 1 then Dec(Day)
                else Day := DaysInMonth(Year, Month);
              end;
            end;
      'H', 'R':
            begin
              if Up then begin
                if Hour < 23 then Inc(Hour) else Hour := 0;
              end else begin
                if Hour > 0 then Dec(Hour) else Hour := 23;
              end;
            end;
      'N':  begin
              if Off = 1 then begin
                if Up then begin
                  if Minute < 59 then Inc(Minute) else Minute := 0;
                end else begin
                  if Minute > 0 then Dec(Minute) else Minute := 59;
                end
              end else begin
                if Up then begin
                  if Minute < 50 then Inc(Minute, 10) else Dec(Minute, 50);
                end else begin
                  if Minute > 9 then Dec(Minute, 10) else Inc(Minute, 50);
                end;
              end;
            end;
      'S':  begin
              if Off = 1 then begin
                if Up then begin
                  if Second < 59 then Inc(Second) else Second := 0;
                end else begin
                  if Second > 0 then Dec(Second) else Second := 59;
                end
              end else begin
                if Up then begin
                  if Second < 50 then Inc(Second, 10) else Dec(Second, 50);
                end else begin
                  if Second > 9 then Dec(Second, 10) else Inc(Second, 50);
                end;
              end;
            end;
      'A', 'B':
            begin
              if Hour < 12 then Inc(Hour, 12) else Dec(Hour, 12);
            end;
      'W':  begin
              if Up then begin
                if Day < DaysInMonth(Year, Month) then Inc(Day)
                else begin
                  Day := 1;
                  if Month < 12 then Inc(Month)
                  else begin
                    Month := 1;
                    Inc(Year);
                  end;
                end;
              end else begin
                if Day > 1 then Dec(Day)
                else begin
                  if Month > 1 then Dec(Month)
                  else begin
                    Month := 12;
                    Dec(Year);
                  end;
                  Day := DaysInMonth(Year, Month);
                end;
              end;
            end;
    end;
    try
      SetDateTime(EncodeDate(Year, Month, Day) + EncodeTime(Hour, Minute, Second, Millis));
    except
      on EConvertError do
        MessageBeep(0);
    end;
    SelStart := Fld - 1;
    case UpCase(FTemplate[Fld]) of
      'N', 'S': SelLength := Ext - Off + 1;
      else      SelLength := Ext;
    end;
  end;
end;

  { Translate FormatDateTime format string (loosely interpreted) into an edit
    mask and template.  The template parallels the edit string space for space
    with characters that describe what goes in each position.  Days of the week
    are inserted in the mask; you can't edit them. }
procedure TCustomDateTimeEdit.UpdateFormat(WeekDay: Word);
var
  AppendLevel: Integer;

  procedure MeasureSystemStrings;
  var
    I: Integer;
  begin
    FLongestAMPMString := Length(TimeAMString);
    if FLongestAMPMString < Length(TimePMString) then
      FLongestAMPMString := Length(TimePMString);
    FLongestDayName := 0;
    for I := 1 to 7 do begin
      if FLongestDayName < Length(LongDayNames[I]) then
        FLongestDayName := Length(LongDayNames[I]);
    end;
    FLongestMonthName := 0;
    for I := 1 to 12 do begin
      if FLongestMonthName < Length(LongMonthNames[I]) then
        FLongestMonthName := Length(LongMonthNames[I]);
    end;

  end;

  procedure AppendFormat(const Format: String);
  var
    FirstChar: Char;
    PrevToken: Char;
    ThisToken: Char;
    QuoteChar: Char;
    Count: Integer;
    Clock12: Boolean;
    Pos: Integer;
    Len: Integer;
    I: Integer;

    function Tokenize(C: Char): Char;
    begin
      Result := UpCase(C);
      if (Result = 'M') and (PrevToken = 'H') then
        Result := 'N';
    end;

    procedure GetCount;
    var
      I: Integer;
    begin
      I := Pos - 1;
      while (Pos <= Len) and (Tokenize(Format[Pos]) = ThisToken) do
        Inc(Pos);
      Count := Pos - I;
    end;

    procedure AppendLiteral(const S: String; TmpChar: Char; Separators: Boolean);
    var
      I: Integer;
    begin
      for I := 1 to Length(S) do begin
        if (System.Pos(S[I], '!<>\LlAaCc09#;_') > 0) or (Separators and
            (System.Pos(S[I], ':/') > 0))  then
          AppendStr(FMask, '\');
        AppendStr(FMask, S[I]);
        AppendStr(FTemplate, TmpChar);
      end;
    end;

  begin
    { Prevent runaway recursive calls for C, T, and DDDDD... formats. }
    if AppendLevel < 2 then begin
      Inc(AppendLevel);
      Pos := 1;
      Len := Length(Format);
      PrevToken := #0;
      while Pos <= Len do begin
        FirstChar := Format[Pos];
        Inc(Pos);
{$IFDEF WIN32}
        if FirstChar in LeadBytes then begin
          if Pos = Len then break;
          Inc(Pos);
          PrevToken := #0;
          continue;
        end;
{$ENDIF}
        ThisToken := Tokenize(FirstChar);
        case ThisToken of
          'Y':  begin
                  GetCount;
                  if (Count <= 2) and not FFourDigitYear then begin
                    AppendStr(FMask, '99');
                    AppendStr(FTemplate, 'YY');
                  end else begin
                    AppendStr(FMask, '9999');
                    AppendStr(FTemplate, 'YYYY');
                  end;
                  PrevToken := ThisToken;
                end;
{$IFDEF WIN32}
          'G':  begin
                  GetCount;
                  { Era indicator, not supported. }
                  PrevToken := ThisToken;
                end;
          'E':  begin
                  GetCount;
                  { Year string (?), not supported. }
                  PrevToken := ThisToken;
                end;
{$ENDIF}
          'M':  begin
                  GetCount;
                  if Count in [1, 2] then begin
                    AppendStr(FMask, '99');
                    if Count = 1 then AppendStr(FTemplate, 'mm')
                    else              AppendStr(FTemplate, 'MM');
                  end else if Count = 3 then begin
                    AppendStr(FMask, 'aaa');
                    AppendStr(FTemplate, 'MMM');
                  end else begin
                    AppendStr(FMask, StringOfChar('a', FLongestMonthName));
                    AppendStr(FTemplate, StringOfChar('M', FLongestMonthName));
                  end;
                  PrevToken := ThisToken;
                end;
          'D':  begin
                  GetCount;
                  if Count in [1, 2] then begin
                    AppendStr(FMask, '99');
                    if Count = 1 then AppendStr(FTemplate, 'dd')
                    else              AppendStr(FTemplate, 'DD');
                  end else if Count = 3 then begin
                    if WeekDay > 0 then
                      AppendLiteral(Pad(ShortDayNames[WeekDay], 3), 'W', true)
                    else begin
                      AppendStr(FMask, '   ');
                      AppendStr(FTemplate, 'WWW');
                    end;
                  end else if Count = 4 then begin
                    if WeekDay > 0 then
                      AppendLiteral(Pad(LongDayNames[Weekday], FLongestDayName),
                                    'W', true)
                    else begin
                      AppendStr(FMask, StringOfChar(' ', FLongestDayName));
                      AppendStr(FTemplate, StringOfChar('W', FLongestDayName));
                    end;
                  end else if Count = 5 then begin
                    AppendFormat(ShortDateFormat);
                  end else begin
                    AppendFormat(LongDateFormat);
                  end;
                  PrevToken := ThisToken;
                end;
          'H':  begin
                  GetCount;
                  Clock12 := false;
                  QuoteChar := #0;
                  I := Pos;
                  while I <= Len do begin
{$IFDEF WIN32}
                    if Format[I] in LeadBytes then begin
                      Inc(I);
                      if I = Len then break;
                    end;
{$ENDIF}
                    case Format[I] of
                      'A', 'a':   begin
                                    if QuoteChar = #0 then begin
                                      if ((CompareText(Copy(Format, I, 5), 'AM/PM') = 0) or
                                          (CompareText(Copy(Format, I, 3), 'A/P'  ) = 0) or
                                          (CompareText(Copy(Format, I, 4), 'AMPM' ) = 0)) then
                                        Clock12 := true;
                                      break;
                                    end;
                                  end;
                      'H', 'h':   break;
                      '''', '"':  begin
                                    if Format[I] = QuoteChar then QuoteChar := #0
                                    else QuoteChar := Format[I];
                                  end;
                    end;
                    Inc(I);
                  end;
                  AppendStr(FMask, '99');
                  if Clock12 then begin
                    if Count = 1 then AppendStr(FTemplate, 'hh')
                    else              AppendStr(FTemplate, 'HH');
                  end else begin
                    if Count = 1 then AppendStr(FTemplate, 'rr')
                    else              AppendStr(FTemplate, 'RR');
                  end;
                  PrevToken := ThisToken;
                end;
          'N':  begin
                  GetCount;
                  AppendStr(FMask, '99');
                  if Count = 1 then AppendStr(FTemplate, 'nn')
                  else              AppendStr(FTemplate, 'NN');
                  PrevToken := ThisToken;
                end;
          'S':  begin
                  GetCount;
                  AppendStr(FMask, '99');
                  if Count = 1 then AppendStr(FTemplate, 'ss')
                  else              AppendStr(FTemplate, 'SS');
                  PrevToken := ThisToken;
                end;
          'T':  begin
                  GetCount;
                  if Count = 1 then AppendFormat(ShortTimeFormat)
                  else              AppendFormat(LongTimeFormat);
                  PrevToken := ThisToken;
                end;
          'A':  begin
                  I := Pos - 1;
                  if CompareText(Copy(Format, I, 5), 'AM/PM') = 0 then begin
                    if Format[I] = 'A' then begin
                      AppendStr(FMask, '>l');
                      AppendStr(FTemplate, 'A');
                    end else begin
                      AppendStr(FMask, '<l');
                      AppendStr(FTemplate, 'a');
                    end;
                    if Format[I+1] = 'M' then begin
                      AppendStr(FMask, '>l');
                      AppendStr(FTemplate, 'A');
                    end else begin
                      AppendStr(FMask, '<l');
                      AppendStr(FTemplate, 'a');
                    end;
                    AppendStr(FMask, '<>');
                    Inc(Pos, 4);
                  end else if CompareText(Copy(Format, I, 3), 'A/P') = 0 then begin
                    if Format[I] = 'A' then begin
                      AppendStr(FMask, '>l');
                      AppendStr(FTemplate, 'A');
                    end else begin
                      AppendStr(FMask, '<l');
                      AppendStr(FTemplate, 'a');
                    end;
                    AppendStr(FMask, '<>');
                    Inc(Pos, 2);
                  end else if CompareText(Copy(Format, I, 4), 'AMPM') = 0 then begin
                    AppendStr(FMask, StringOfChar('l', FLongestAMPMString));
                    AppendStr(FTemplate, StringOfChar('b', FLongestAMPMString));
                    Inc(Pos, 3);
{$IFDEF WIN32}
                  end else if CompareText(Copy(Format, I, 4), 'AAAA') = 0 then begin
                    { What about AAAAA...? }
                    if WeekDay > 0 then
                      AppendLiteral(Pad(LongDayNames[Weekday], FLongestDayName),
                                    'W', true)
                    else begin
                      AppendStr(FMask, StringOfChar(' ', FLongestDayName));
                      AppendStr(FTemplate, StringOfChar('W', FLongestDayName));
                    end;
                    Inc(Pos, 3);
                  end else if CompareText(Copy(Format, I, 3), 'AAA') = 0 then begin
                    if WeekDay > 0 then
                      AppendLiteral(Pad(ShortDayNames[WeekDay], 3), 'W', true)
                    else begin
                      AppendStr(FMask, '   ');
                      AppendStr(FTemplate, 'WWW');
                    end;
                    Inc(Pos, 2);
{$ENDIF}
                  end else begin
                    AppendStr(FMask, FirstChar);
                    AppendStr(FTemplate, ' ');
                  end;
                  PrevToken := ThisToken;
                end;
          'C':  begin
                  GetCount;
                  AppendFormat(ShortDateFormat);
                  AppendFormat(' ');
                  AppendFormat(LongTimeFormat);
                  PrevToken := ThisToken;
                end;
          '''',
          '"':  begin
                  I := Pos;
                  while (Pos <= Len) and (Format[Pos] <> FirstChar) do begin
{$IFDEF WIN32}
                    if (Format[Pos] in LeadBytes) then begin
                      Inc(Pos);
                      if Pos = Len then break;
                    end;
{$ENDIF}
                    Inc(Pos);
                  end;
                  AppendLiteral(Copy(Format, I, Pos - I), ' ', true);
                  if Format[Pos] = FirstChar then
                    Inc(Pos);
                end;
          else  AppendLiteral(FirstChar, ' ', false);
        end;
      end;
      Dec(AppendLevel);
    end;
  end;

begin { UpdateFormat }
  FMask := '';
  FTemplate := '';
  AppendLevel := 0;
  MeasureSystemStrings;
  if FDateFormat = '' then AppendFormat('C')
  else AppendFormat(FDateFormat);
  EditMask := FMask + ';0;' + FBlankChar;
end;

{$IFDEF BTNKIND}
function TCustomDateTimeEdit.GetButtonKind: TSpinButtonKind;
begin
  if NewStyleControls then Result := FButtonKind
  else Result := bkDiagonal;
end;

procedure TCustomDateTimeEdit.SetButtonKind(Value: TSpinButtonKind);
var
  OldKind: TSpinButtonKind;
begin
  OldKind := FButtonKind;
  FButtonKind := Value;
  if OldKind <> GetButtonKind then begin
    RecreateButton;
    ResizeButton;
    SetEditRect;
  end;
end;
{$ENDIF}

{$IFDEF WIN32}
procedure TCustomDateTimeEdit.UpDownClick(Sender: TObject; Button: TUDBtnType);
begin
  if TabStop and CanFocus then SetFocus;
  UpDownButtonClick(Button = btNext);
end;
{$ENDIF}

function TCustomDateTimeEdit.GetButtonWidth: Integer;
begin
  Result := 0;
{$IFDEF RXLIB}
  if Assigned(FButton) then Result := FButton.Width;
{$ENDIF}
{$IFDEF WIN32}
  if Assigned(FUpDown) then Result := FUpDown.Width;
{$ENDIF}
end;

procedure TCustomDateTimeEdit.RecreateButton;
begin
{$IFDEF RXLIB}
  FButton.Free;
  FButton := nil;
  FBtnWindow.Free;
  FBtnWindow := nil;
{$ENDIF}
{$IFDEF WIN32}
  FUpDown.Free;
  FUpDown := nil;
{$ENDIF}
  if not FSpinButton then exit;
{$IFDEF WIN32}
{$IFDEF BTNKIND}
  if GetButtonKind = bkStandard then
{$ENDIF}
  begin
    FUpDown := TDateTimeUpDown.Create(Self);
    with TDateTimeUpDown(FUpDown) do begin
      Visible := true;
      SetBounds(0, 0, GetSystemMetrics(SM_CXVSCROLL), Self.Height);
      Align := alRight;
      Parent := Self;
      OnClick := UpDownClick;
    end;
    exit;
  end;
{$ENDIF}
{$IFDEF RXLIB}
  FBtnWindow := TWinControl.Create(Self);
  FBtnWindow.Visible := true;
  FBtnWindow.Parent := Self;
  FBtnWindow.SetBounds(0, 0, Height, Height);
  FButton := TRxSpinButton.Create(Self);
  FButton.Visible := true;
  FButton.Parent := FBtnWindow;
  FButton.FocusControl := Self;
  FButton.OnTopClick := UpClick;
  FButton.OnBottomClick := DownClick;
  FButton.SetBounds(0, 0, FBtnWindow.Width, FBtnWindow.Height);
{$ENDIF}
end;

procedure TCustomDateTimeEdit.ResizeButton;
begin
{$IFDEF WIN32}
  if Assigned(FUpDown) then begin
    FUpDown.Width := GetSystemMetrics(SM_CXVSCROLL);
    FUpDown.Align := alRight;
  end;
{$ENDIF}
{$IFDEF RXLIB}
  if Assigned(FBtnWindow) then begin
{$IFDEF WIN32}
    if NewStyleControls and Ctl3D and (BorderStyle = bsSingle) then
      FBtnWindow.SetBounds(Width - Height - 1, -1, Height - 3, Height - 3)
    else
{$ENDIF}
      FBtnWindow.SetBounds(Width - Height, 0, Height, Height);
    FButton.SetBounds(0, 0, FBtnWindow.Width, FBtnWindow.Height);
  end;
{$ENDIF RXLIB}
end;

procedure TCustomDateTimeEdit.SetEditRect;
var
  R: TRect;
begin
  SetRect(R, 0, 0, ClientWidth - GetButtonWidth - 2, ClientHeight + 1);
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@R));
end;

procedure TCustomDateTimeEdit.UpClick(Sender: TObject);
begin
  UpDownButtonClick(true);
end;

procedure TCustomDateTimeEdit.DownClick(Sender: TObject);
begin
  UpDownButtonClick(false);
end;

procedure TCustomDateTimeEdit.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
{$IFDEF RXLIB}
  if Assigned(FButton) then FButton.Enabled := Enabled;
{$ENDIF}
{$IFDEF WIN32}
  if Assigned(FUpDown) then FUpDown.Enabled := Enabled;
{$ENDIF}
end;

procedure TCustomDateTimeEdit.CMExit(var Message: TCMExit);
begin
  inherited;
  if Valid and not Blank then begin
    FChanging := true;
    try
      SetDateTime(DateTime);
    finally
      FChanging := false;
    end;
  end;
end;

procedure TCustomDateTimeEdit.WMCut(var Message: TWMCut);
begin
  if FEditorEnabled then inherited;
end;

procedure TCustomDateTimeEdit.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  inherited;
  { ES_MULTILINE controls normally trap Enter, but we don't want that. }
  Message.Result := Message.Result and not DLGC_WANTALLKEYS;
end;

procedure TCustomDateTimeEdit.WMPaste(var Message: TWMPaste);
begin
  if FEditorEnabled then inherited;
end;

procedure TCustomDateTimeEdit.WMSize(var Message: TWMSize);
begin
  inherited;
  ResizeButton;
  SetEditRect;
end;

procedure TCustomDateTimeEdit.Evaluate;
var
  Year:   Word;
  Month:  Word;
  Day:    Word;
  Hour:   Word;
  Minute: Word;
  Second: Word;
  Millis: Word;
  MaskText: String;
  Count: Integer;
  Len: Integer;
  S: String;
  P: Integer;
  Q: Integer;
  I: Integer;
begin
  if not FEvaluated then begin
    FEvaluated := true;
    try
      DecodeDate(FDateTime, Year, Month, Day);
      DecodeTime(FDateTime, Hour, Minute, Second, Millis);
      FBlank := Trim(Text) = '';
      if not FBlank then begin
        MaskText := FormatMaskText(FMask + ';0; ', Text);
        Len := Length(FTemplate);
        P := 1;
        while P <= Len do begin
          Q := P;
          while (P <= Len) and (UpCase(FTemplate[P]) = UpCase(FTemplate[Q])) do Inc(P);
          Count := P - Q;
          S := Trim(Copy(MaskText, Q, Count));
          case UpCase(FTemplate[Q]) of
            'Y':  begin
                    Year := StrToInt(S);
                    if Length(S) <= 2 then begin
                      Inc(Year, FEpoch - FEpoch mod 100);
                      if Year < FEpoch then
                        Inc(Year, 100);
                    end;
                  end;
            'M':  begin
                    if (Length(S) > 0) and (S[1] in ['0'..'9']) then
                      Month := StrToInt(S)
                    else begin
                      Month := 0;
                      for I := 1 to 12 do begin
                        if AnsiCompareText(S, ShortMonthNames[I]) = 0 then
                          Month := I;
                        if AnsiCompareText(S, LongMonthNames[I]) = 0 then
                          Month := I;
                      end;
                    end;
                  end;
            'D':  Day := StrToInt(S);
            'H':  begin
                    Hour := StrToInt(S);
                    if Hour = 12 then Hour := 0;
                    if Hour > 12 then Hour := 24;
                  end;
            'R':  Hour := StrToInt(S);
            'N':  Minute := StrToInt(S);
            'S':  Second := StrToInt(S);
            'A':  begin
                    if Hour < 24 then begin
                      if (Length(S) > 0) and (S[1] in ['A', 'a']) then
                        Hour := Hour mod 12
                      else if (Length(S) > 0) and (S[1] in ['P', 'p']) then
                        Hour := Hour mod 12 + 12
                      else Hour := 24;
                    end;
                  end;
            'B':  begin
                    if Hour < 24 then begin
                      if AnsiCompareText(S, TimeAMString) = 0 then
                        Hour := Hour mod 12
                      else if AnsiCompareText(S, TimePMString) = 0 then
                        Hour := Hour mod 12 + 12
                      else Hour := 24;
                    end;
                  end;
          end;
        end;
      end;
      if (Month in [1..12]) and (Day in [1..31]) and (Hour in [0..23]) and
         (Minute in [0..59]) and (Second in [0..59]) then begin
        FDateTime := EncodeDate(Year, Month, Day) +
                     EncodeTime(Hour, Minute, Second, Millis);
        FValid := true;
      end else
        FValid := false;
    except
      on EConvertError do
        FValid := false;
    end;
  end;
end;

function TCustomDateTimeEdit.FormatHasWeekDay: Boolean;
begin
  Result := Pos('W', FTemplate) > 0;
end;

function TCustomDateTimeEdit.GetBlank: Boolean;
begin
  Evaluate;
  Result := FBlank;
end;

function TCustomDateTimeEdit.GetDateTime: TDateTime;
begin
  Evaluate;
  Result := FDateTime;
end;

function TCustomDateTimeEdit.GetValid: Boolean;
begin
  Evaluate;
  Result := FValid;
end;

procedure TCustomDateTimeEdit.SetBlank(Value: Boolean);
begin
  if GetBlank <> Value then begin
    if Value then Clear
    else SetDateTime(FDateTime);
  end;
end;

procedure TCustomDateTimeEdit.SetBlankChar(Value: Char);
begin
  if FBlankChar <> Value then begin
    Evaluate;
    FBlankChar := Value;
    FChanging := true;
    try
      EditMask := FMask + ';0;' + FBlankChar;
      if not FBlank then
        SetDateTime(FDateTime);
    finally
      FChanging := false;
    end;
  end;
end;

procedure TCustomDateTimeEdit.SetDateFormat(const Value: String);
var
  WeekDay: Word;
begin
  if FDateFormat <> Value then begin
    Evaluate;
    if FBlank then WeekDay := 0
    else WeekDay := DayOfWeek(FDateTime);
    FDateFormat := Value;
    FChanging := true;
    try
      UpdateFormat(WeekDay);
      if not FBlank then
        SetDateTime(FDateTime);
    finally
      FChanging := false;
    end;
  end;
end;

procedure TCustomDateTimeEdit.SetDateTime(Value: TDateTime);
var
  Year:   Word;
  Month:  Word;
  Day:    Word;
  Hour:   Word;
  Minute: Word;
  Second: Word;
  Millis: Word;
  Count: Integer;
  Temp: Integer;
  STmp: String;
  Len: Integer;
  S: String;
  P: Integer;
  Q: Integer;

  procedure AppendString(const Value: String; Width: Integer);
  begin
    AppendStr(S, Pad(Value, Width));
  end;

  procedure AppendNumber(Value: Integer; Width: Integer; ZeroPad: Boolean);
  begin
    if ZeroPad then AppendStr(S, Format('%*.*d', [Width, Width, Value]))
    else            AppendStr(S, Format('%*d', [Width, Value]));
  end;

begin
  S := '';
  if FormatHasWeekDay then
    UpdateFormat(DayOfWeek(Value));
  DecodeDate(Value, Year, Month, Day);
  DecodeTime(Value, Hour, Minute, Second, Millis);
  Len := Length(FTemplate);
  P := 1;
  while P <= Len do begin
    Q := P;
    while (P <= Len) and (UpCase(FTemplate[P]) = UpCase(FTemplate[Q])) do Inc(P);
    Count := P - Q;
    case UpCase(FTemplate[Q]) of
      'Y':  begin
              if Count <= 2 then AppendNumber(Year mod 100, Count, true)
              else               AppendNumber(Year, Count, true);
            end;
      'M':  begin
              case Count of
                1, 2: AppendNumber(Month, Count, FTemplate[Q] = 'M');
                3:    AppendStr(S, Format('%-3s', [ShortMonthNames[Month]]));
                else  AppendStr(S, Format('%-*s', [Count, LongMonthNames[Month]]));
              end;
            end;
      'D':  AppendNumber(Day, Count, FTemplate[Q] = 'D');
      'H':  begin
              Temp := Hour mod 12;
              if Temp = 0 then Temp := 12;
              AppendNumber(Temp, Count, FTemplate[Q] = 'H');
            end;
      'R':  AppendNumber(Hour, Count, FTemplate[Q] = 'R');
      'N':  AppendNumber(Minute, Count, FTemplate[Q] = 'N');
      'S':  AppendNumber(Second, Count, FTemplate[Q] = 'S');
      'A':  begin
              if Hour < 12 then STmp := 'a'
              else              STmp := 'p';
              if FTemplate[Q] = 'A' then  STmp := UpperCase(STmp);
              AppendStr(S, STmp);
              if Count > 1 then begin
                if FTemplate[Q+1] = 'A' then STmp := 'M'
                else                         STmp := 'm';
                AppendStr(S, STmp);
              end;
            end;
      'B':  begin
              if Hour < 12 then STmp := TimeAMString
              else              STmp := TimePMString;
              AppendString(STmp, Count);
            end;
    end;
  end;
  if FDateTime <> Value then begin
    FDateTime := Value;
    Change;
  end;
  if Text <> S then
    Text := S;
end;

procedure TCustomDateTimeEdit.SetFourDigitYear(Value: Boolean);
var
  WeekDay: Word;
begin
  if FFourDigitYear <> Value then begin
    Evaluate;
    if FBlank then WeekDay := 0
    else WeekDay := DayOfWeek(FDateTime);
    FFourDigitYear := Value;
    FChanging := true;
    try
      UpdateFormat(WeekDay);
      if not FBlank then
        SetDateTime(FDateTime);
    finally
      FChanging := false;
    end;
  end;
end;

procedure TCustomDateTimeEdit.SetSpinButton(Value: Boolean);
begin
{$IFDEF SPIN}
  if FSpinButton <> Value then begin
    FSpinButton := Value;
    RecreateButton;
    ResizeButton;
    SetEditRect;
  end;
{$ENDIF}
end;


{ TDBDateTimeEdit }

constructor TDBDateTimeEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := true;
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
end;

destructor TDBDateTimeEdit.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TDBDateTimeEdit.Change;
begin
  if not FChanging and Assigned(FDataLink) then
    FDataLink.Modified;
  inherited Change;
end;

function TDBDateTimeEdit.EditCanModify: Boolean;
begin
  Result := FDataLink.Edit;
end;

procedure TDBDateTimeEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if not ReadOnly and ((Key = VK_DELETE) or ((Key = VK_INSERT) and
                       (ssShift in Shift))) then
    FDataLink.Edit;
end;

procedure TDBDateTimeEdit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  case Key of
    ^H, ^V, ^X, #32..#255:
          FDataLink.Edit;
    #27:  begin
            Reset;
            Key := #0;
          end;
  end;
end;

procedure TDBDateTimeEdit.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and Assigned(FDataLink) and
     (AComponent = DataSource) then
    DataSource := nil;
end;

procedure TDBDateTimeEdit.Reset;
begin
  FDataLink.Reset;
  if AutoSelect then SelectAll;
end;

procedure TDBDateTimeEdit.UpDownButtonClick(Up: Boolean);
begin
  FDataLink.Edit;
  inherited UpDownButtonClick(Up);
end;

procedure TDBDateTimeEdit.DataChange(Sender: TObject);
begin
  if (FDataLink.Field <> nil) then begin
    if FDataLink.Field.IsNull then Clear
    else DateTime := FDataLink.Field.AsDateTime;
  end else begin
    DateTime := 0;
    Clear;
  end;
end;

procedure TDBDateTimeEdit.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not FDataLink.Editing;
end;

procedure TDBDateTimeEdit.UpdateData(Sender: TObject);
begin
  ValidateEdit;
  Validate;
  if Blank then FDataLink.Field.Clear
  else FDataLink.Field.AsDateTime := DateTime;
end;

procedure TDBDateTimeEdit.CMEnter(var Message: TCMEnter);
begin
  FDataLink.Reset;
  inherited;
end;

procedure TDBDateTimeEdit.CMExit(var Message: TCMExit);
begin
  try
    Validate;
    FDataLink.UpdateRecord;
  except
    if CanFocus then SetFocus;
    if AutoSelect then SelectAll;
    raise;
  end;
  FDataLink.Reset;
  inherited;
end;

{$IFDEF WIN32}
procedure TDBDateTimeEdit.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;
{$ENDIF}

procedure TDBDateTimeEdit.WMCut(var Message: TMessage);
begin
  if EditorEnabled then begin
    FDataLink.Edit;
    inherited;
  end;
end;

procedure TDBDateTimeEdit.WMPaste(var Message: TMessage);
begin
  if EditorEnabled then begin
    FDataLink.Edit;
    inherited;
  end;
end;

function TDBDateTimeEdit.GetDataField: String;
begin
  Result := FDataLink.FieldName;
end;

function TDBDateTimeEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

function TDBDateTimeEdit.GetField: TField;
begin
  Result := FDataLink.Field;
end;

function TDBDateTimeEdit.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TDBDateTimeEdit.SetDataField(const Value: String);
begin
  FDataLink.FieldName := Value;
end;

procedure TDBDateTimeEdit.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
{$IFDEF WIN32}
  if Assigned(Value) then
    Value.FreeNotification(Self);
{$ENDIF}
end;

procedure TDBDateTimeEdit.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

end.
