{************************************************
 *                                              *
 *                TCP Device Unit               *
 *                                              *
 *  Interfacing the WinSock Library for Pascal  *
 *                                              *
 *  Copyright (c) 1998 Robert Morewood          *
 *                    (morewood@planeteer.com)  *
 ************************************************}

Unit TCP;
{A simplified interface for creating TCP (Transport Control
 Protocol) connections between applications in a network
 using the WinSock library (part of most internet software
 packages for Windows).}

Interface

Const Socket_Error   = -1;
      Invalid_Socket = 65535 {NOT(0)};

Type
{Data supplied by the WinSock library.}
   WinSock_Data_Pointer = ^WinSock_Data_Type;
   WinSock_Data_Type = record
       Version : word;
       HighVersion : word;
       Description : array[0..256] of char;
       SystemStatus : array[0..128] of char;
       MaxSockets : integer;
       MaxUDPPacketSize : integer;
       VendorInfo : PChar;
   end;
{Status codes returned by the CheckTCP function.}
   TCP_Status=(
     TCP_Error,    {Something went wrong at the WinSock level.
                    The WinSock Error code is in TCP_Handle^.WinSockError}
     TCP_Ready,    {Data from a remote application is ready to be read,
                    ReadTCP will return immediately.}
     TCP_Active,   {Connected to a remote application,
                    but no data is currently waiting to be read.}
     TCP_Listening,{Waiting for a remote application to connect.}
     TCP_Closed    {The remote application closed the connection.});
{Data about the connection.
 Socket and ListeningSocket are used to communicate
 with the WinSock library.  DO NOT CHANGE!
 The rest are read-only for the information of the application.}
   TCP_Record=Record
       Socket        : Word;   {Active Connection Information} 
       LocalPort     : Word;
       LocalAddress  : LongInt;
       LocalName     : String;
       RemotePort    : Word;
       RemoteAddress : LongInt;
       RemoteName    : String;
       ListeningSocket : Word; {Waiting for a remote connection.}
       WinSockData  : WinSock_Data_Pointer;
       WinSockError : Integer;
       Status       : TCP_Status;
   end;
{** Every application using this unit     **
 ** must declare a TCP_Handle variable    **
 ** to pass to the TCP control functions. **} 
   TCP_Handle=^TCP_Record;

{A list of status strings to make status printout easy:
  WriteLn(StatusName[CheckTCP(TCP_Handle)]); }
Const StatusName:Array[TCP_Error..TCP_Closed] of String=
      ('TCP_Error','TCP_Ready','TCP_Active','TCP_Listening','TCP_Closed');

{***********************
 *  PUBLIC FUNCTIONS   *
 ***********************}
{The boolean functions in this list all return TRUE unless something
 unexpected happens and the socket layer generates an error.
 In that case the WinSock error code is stored in TCP_Handle^.WinSockError}

Function OpenTCP(Var Handle:TCP_Handle; Port:Word; Address:String):Boolean;
{Opens a TCP connection to the specified port at the given address.
 Address is an ascii string such as:  'south.sd41.bc.ca'
 A blank address string starts listening for a remote application
 to connect on the given port.  A connection is created when one
 application listens on a 'well-known' port and then second application
 connects to that port at the address of the first application.}

Function ReadTCP(Var Handle:TCP_Handle; Var Data:String):Boolean;
{Reads data from the specified TCP Socket.  Like keyboard reads, this
 function does not return until data is received or the connection is
 closed.  Use CheckTCP first!
 A Blank string will be returned if and ONLY if the connection closes.}

Function CheckTCP(Var Handle:TCP_Handle):TCP_Status;
{Checks to see if data is waiting to be read.
 Can return any of the TCP_Status codes as appropriate (see above)}

Function WriteTCP(Var Handle:TCP_Handle; Data:String):Boolean;
{Send data to the TCP Socket.  Does NOT add Carriage Return or LineFeed
 characters.  Most applications will require that these be added to the
 data (+#13+#10).}

Function CloseTCP(Var Handle:TCP_Handle):Integer;
{Shuts down the socket and releases associated memory.
 Returns the last Socket Error code (zero if all went well).}


Implementation
{You don't NEED to read this,
 unless you want to improve HOW this unit works.
 Please let me know of any improvements which you do make
 so that I and my students may benefit as well!
 NOTE: The WinSock functions used here have been renamed
 to aid my understanding of what they do.}

Function GetLastErrorFromWinSock
                            :Integer; far; external 'WINSOCK' index 111;

Type PSocketAddress=^SocketAddress;
     SocketAddress = record
       Family  : Word; {Always Internet=2}
       Port    : Word;
       Address : LongInt;
       Pad     : Array[0..7] of Char;
     end;
     PInteger=^Integer;
     PLongInt=^LongInt;

Procedure CopyPChar2String(Source:Pchar;Var Destination:String);
{Copy Window-style strings returned by WinSock into Pascal-style strings.}
  Var Count:Byte;
  Begin
    Count:=0;
    While (Source^>#0) and (Count<255) do begin
      Inc(Count);
      Destination[Count]:=Source^;
      Inc(Source);
    end;
    Destination[0]:=Chr(Count);
  end;
            
{*** OPEN_TCP Section ***}

Function WinSock_Startup(Version : word; WinSock_Data : WinSock_Data_Pointer)
                            :Integer; far; external 'WINSOCK' index 115;
Const Internet=2;TCP_Type=1;Protocol=0;
Function Socket(Network_Type, Socket_Type, Protocol_Type : Integer)
                               :Word; far; external 'WINSOCK' index 23;
Const Socket_Level=-1; ReUsePorts=4; OutOfBand_Inline=256; On:Char=#255;
Function SetSocketOption(Socket:Word;Level,Option:Integer;Value:PChar;
               Length:Integer):Integer;far;external 'WINSOCK' index 21;
Function Bind(Socket:Word; Address:PSocketAddress; AddressLength:Integer)
                            :Integer; far; external 'WINSOCK' index 2;
Function GetSocketAddress(Socket:Word; Address:PSocketAddress; AddLen:PInteger)
                            :Integer; far; external 'WINSOCK' index 6;
Function Listen(Socket:Word; QueueLength:integer)
                            :Integer; far; external 'WINSOCK' index 13;
Function Convert_to_Network_Word(Number : Word)
                               :Word; far; external 'WINSOCK' index 9;
Function Convert_to_Host_Word(Number : Word)
                               :Word; far; external 'WINSOCK' index 15;
Function Convert_to_Host_Long(Number : LongInt)
                            :LongInt; far; external 'WINSOCK' index 14;
Function GetHostName(NameBuffer:PChar; Length:Integer)
                            :Integer; far; external 'WINSOCK' index 57;
Type HostPointer=^HostType;
     HostType = record
              Name : PChar;
              Aliases : ^PChar;      {Null Terminated List}
              Addrtype : integer;    {Always Internet}
              length : integer;      {Always 4}
              addresses : ^PLongInt; {Null Terminated List}
            end;
Function GetHostByName(Name : PChar)
                        :HostPointer; far; external 'WINSOCK' index 52;
Function Connect(Socket:Word; Address:PSocketAddress; AddressLength:Integer)
                            :Integer; far; external 'WINSOCK' index 4;

Function OpenTCP(Var Handle:TCP_Handle; Port:Word; Address:String):Boolean;
{Sets up a TCP connection to the address/port given 
 or listens for a connection on the given port if address is blank.}
Var Location:SocketAddress;
    Lngth   :Integer;
    Host    :HostPointer;
Begin
  {Initialize Socket}
  OpenTCP:=False;
  GetMem(Handle,SizeOf(TCP_Record));
  If Handle<>Nil Then Begin
   GetMem(Handle^.WinSockData,SizeOf(WinSock_Data_Type));
   If Handle^.WinSockData<>Nil Then
    If WinSock_Startup($0101,Handle^.WinSockData)<>Socket_Error Then Begin
     If GetHostName(@Handle^.LocalName[1],255)=0 then begin
      Lngth:=1;
      While (Handle^.LocalName[Lngth]>#0) and (Lngth<256) do Inc(Lngth);
      Handle^.LocalName[0]:=Chr(Lngth-1);
      Host:=GetHostByName(@Handle^.LocalName[1]);
      Handle^.LocalAddress:=Convert_To_Host_Long(Host^.Addresses^^);
     end else begin
      Handle^.LocalName:='';
      Handle^.LocalAddress:=0;
     end;
     Handle^.Socket:=Socket(Internet,TCP_Type,Protocol);
     If (Handle^.Socket<>Invalid_Socket) Then Begin
      SetSocketOption(Handle^.Socket,Socket_Level,OutOfBand_Inline,@On,1);
      Location.Family:=Internet;
      Location.Port:=Convert_to_Network_Word(Port);
      If Address=''
       {Listen for any remote connection.}
       Then Begin
        Location.Address:=0;
        Lngth:=16;
        SetSocketOption(Handle^.Socket,Socket_Level,ReUsePorts,@On,1);
        If Bind(Handle^.Socket,@Location,Lngth)<>Socket_Error Then 
         If GetSocketAddress(Handle^.Socket,@Location,@Lngth)<>Socket_Error
          Then Begin
           Handle^.LocalPort:=Convert_to_Host_Word(Location.Port);
           If Location.Address>0 Then
             Handle^.LocalAddress:=Convert_to_Host_Long(Location.Address);
           If Listen(Handle^.Socket,5)<>Socket_Error Then Begin
            OpenTCP:=True;
            Handle^.ListeningSocket:=Handle^.Socket;
            Handle^.Status:=TCP_Listening;
            Handle^.Socket:=Invalid_Socket;
            Handle^.RemoteAddress:=0;
            Handle^.RemotePort:=0;
            Handle^.RemoteName:='';
           End;
          End;        
       End
       {Connect to the given address.}
       Else Begin
        SetSocketOption(Handle^.Socket,Socket_Level,OutOfBand_Inline,@On,1);
        If Length(Address)<255 then Address[Length(Address)+1]:=#0
                               else Address[255]:=#0;
        Host:=GetHostByName(@Address[1]);
        If Host<>Nil then begin
         Location.Address:=Host^.Addresses^^;
         Handle^.RemotePort:=Port;
         Handle^.RemoteAddress:=Convert_To_Host_Long(Location.Address);
         CopyPChar2String(Host^.Name,Handle^.RemoteName);
         If Connect(Handle^.Socket,@Location,16)<>Socket_Error Then begin
          OpenTCP:=True;
          Handle^.Status:=TCP_Active;
          Handle^.ListeningSocket:=Invalid_Socket;
          Lngth:=16;
          If GetSocketAddress(Handle^.Socket,@Location,@Lngth)<>Socket_Error
           Then Begin
                Handle^.LocalPort:=Convert_to_Host_Word(Location.Port);
                Handle^.LocalAddress:=Convert_to_Host_Long(Location.Address);
           end
           Else Handle^.LocalPort:=0;
         End;
        End;
       End;
     End;
    End;
   Handle^.WinSockError:=GetLastErrorFromWinSock;
  End;
End;

Function Accept(Socket:Word;  name:PSocketAddress; namelength:Pinteger)
                            :Integer; far; external 'WINSOCK' index 1;
Function GetHostByAddr(Address:PLongInt;AddressLength,AddressType:Integer)
                        :HostPointer; far; external 'WINSOCK' index 51;

Function AcceptConnection(Var Handle:TCP_Handle):Boolean;
{Set up a connection for a connection request on the Listening socket.}
Var Location:SocketAddress;
    Length:Integer;
    Host:HostPointer;
Begin
  AcceptConnection:=False;
  If Handle^.ListeningSocket<>Invalid_Socket Then Begin
    Length:=16;
    Handle^.Socket:=Accept(Handle^.ListeningSocket,@Location,@Length);
    If Handle^.Socket<>Invalid_Socket Then Begin
      Handle^.RemotePort:=Convert_to_Host_Word(Location.Port);
      Handle^.RemoteAddress:=Convert_to_Host_Long(Location.Address);
      Host:=GetHostByAddr(@Location.Address,4,Internet);
      If Host<>Nil 
        Then CopyPChar2String(Host^.Name,Handle^.RemoteName)
        Else Handle^.RemoteName:='';
      Handle^.Status:=TCP_Active;
      If GetSocketAddress(Handle^.Socket,@Location,@Length)=0
        Then Handle^.LocalAddress:=Convert_to_Host_Long(Location.Address);      
      AcceptConnection:=True;
    End;
  End;
End;

{** READ_TCP Section ***}
Function Receive(Socket : Word; buffer : PChar; len, flags : integer)
                            :Integer; far; external 'WINSOCK' index 16;
Function CloseSocket(Socket: Word)
                            :Integer; far; external 'WINSOCK' index 3;

Function ReadTCP(Var Handle:TCP_Handle; Var Data:String):Boolean;
{Read Data from the TCP connection - Accept a listening connection
 if there is no active connection.  If the connection closes,
 close the socket and go back to listening.}
Var DataLength : Integer; Name:SocketAddress;
Begin
  ReadTCP:=False;
  Data:='';
  If Handle^.Socket=Invalid_Socket Then AcceptConnection(Handle);
  If Not(Handle^.Socket=Invalid_Socket) then Begin
    DataLength := Receive(Handle^.Socket, @Data[1], 255, 0);
    If DataLength = Socket_Error
      Then Handle^.Status:=TCP_Error
      Else Begin
        Data[0]:=Chr(DataLength);
        ReadTCP:=True;
        If DataLength=0 Then Begin
          CloseSocket(Handle^.Socket);
          Handle^.Socket:=Invalid_Socket;
          If Handle^.ListeningSocket=Invalid_Socket
            Then Handle^.Status:=TCP_Closed
            Else Handle^.Status:=TCP_Listening;
        End;
      End
  End;
  Handle^.WinSockError:=GetLastErrorFromWinSock;
End;

{*** CHECK_TCP Section ***}
Type TCP_Set_Pointer=^TCP_set;
     TCP_Set = record
       TCP_count : Word;
       TCP_array : array[0..1] of word;
     end;
     timeval = record
       sec, microsec : longint;
     end;
Const NoWait:TimeVal=(sec:0;microsec:0);
Function Select(Junk:integer; ToRead, ToWrite, InError : TCP_Set_Pointer;
                timeout : timeval):longint; far; external 'WINSOCK' index 18;

Function CheckTCP(Var Handle:TCP_Handle):TCP_Status;
{Checks the TCP connection for readability and closure.
 If their is no active connection it checks the listening
 socket and accepts a new connection if one is pending.}
Var AcceptSet,ReadSet,CloseSet:TCP_set;
Begin
  {If no active socket, check listening socket.
   Accept any waiting connections.}
  If Handle^.Socket=Invalid_Socket Then
    If Handle^.ListeningSocket=Invalid_Socket
      Then Handle^.Status:=TCP_Closed
      Else Begin {Check the status of the listening socket.}
        AcceptSet.TCP_Count:=1;
        AcceptSet.TCP_Array[0]:=Handle^.ListeningSocket;
        AcceptSet.TCP_Array[1]:=Invalid_Socket;
        If Select(1,@AcceptSet,Nil,Nil,NoWait)=Socket_Error
          Then Handle^.Status:=TCP_Error
          Else If AcceptSet.TCP_Count=0
            Then Handle^.Status:=TCP_Listening         
            Else if not AcceptConnection(Handle)
              Then Handle^.Status:=TCP_Error
              Else Handle^.Status:=TCP_Active;
      end;
  If Not (Handle^.Socket=Invalid_Socket) Then Begin
    {Check Active Socket for Readability or Closure.}
    ReadSet.TCP_Count:=1;
    ReadSet.TCP_Array[0]:=Handle^.Socket;
    ReadSet.TCP_Array[1]:=Invalid_Socket;
    CloseSet:=ReadSet;
    If Select(1,@ReadSet,Nil,@CloseSet,NoWait)=Socket_Error
      Then Handle^.Status:=TCP_Error
      Else If ReadSet.TCP_Count=0 
        Then Handle^.Status:=TCP_Active
        Else If CloseSet.TCP_Count=0
          then Handle^.Status:=TCP_Ready
          Else Begin
            CloseSocket(Handle^.Socket);
            Handle^.Socket:=Invalid_Socket;
            If Handle^.ListeningSocket=Invalid_Socket
              Then Handle^.Status:=TCP_Closed
              Else Handle^.Status:=TCP_Listening;
          end;
  end;
  Handle^.WinSockError:=GetLastErrorFromWinSock;
  CheckTCP:=Handle^.Status;
End;

{*** WRITE_TCP Section ***}
Function Send(Socket : Word; buffer : PChar; len, flags : integer)
                             :integer; far; external 'WINSOCK' index 19;

Function WriteTCP(Var Handle:TCP_Handle; Data:String):Boolean;
{Sends data out on the TCP connection.  Maximum of 254 characters!
 If there is a listening socket, then a blank data string will
 close the current connection so that a new one can be accepted.}
Var Address:SocketAddress; AddressLength:Integer;
Begin
  WriteTCP:=False;
  If Handle^.Socket=Invalid_Socket Then AcceptConnection(Handle);
  If Not(Handle^.Socket=Invalid_Socket) then Begin
    If (Data='') and Not(Handle^.ListeningSocket=Invalid_Socket)
      Then Begin
        CloseSocket(Handle^.Socket);
        Handle^.Socket:=Invalid_Socket;
        Handle^.Status:=TCP_Listening;
        WriteTCP:=True;
      end
      Else Begin
        If Length(Data)<255 then Data[Length(Data)+1]:=#0
                            else Data[255]:=#0;
        If Send(Handle^.Socket, @Data[1], Length(Data), 0) = Socket_Error
          Then WriteTCP:=False
          Else WriteTCP:=True;
      end;
  end;
  Handle^.WinSockError:=GetLastErrorFromWinSock;
End;

{*** CLOSE_TCP Section ***}
Function WSA_CleanUp:Integer; far; external 'WINSOCK' index 116;

Function CloseTCP(Var Handle:TCP_Handle):Integer;
{Closes any open sockets and deallocates memory.}
Begin
  If Handle<>Nil Then begin
    If Handle^.Socket<>Invalid_Socket
      Then CloseSocket(Handle^.Socket);
    If Handle^.ListeningSocket<>Invalid_Socket
      Then CloseSocket(Handle^.ListeningSocket);
    If Handle^.WinSockData<>Nil
      Then FreeMem(Handle^.WinSockData,SizeOf(WinSock_Data_Type));
    FreeMem(Handle,SizeOf(TCP_Record));
  End;
  CloseTCP:=GetLastErrorFromWinSock;
  WSA_CleanUp;
End;

End.