/* **************************************************************************
*                                                                           *
*  PCB.CPP                                                                  *
*                                                                           *
*  23-01-97                                                    BUILD:0002   *
*                                                                           *
*  (c) Copyright, 1996-1997 de Daniel Vil i Amill                          *
*                                                                           *
*  Aquest fitxer pot ser utilitzat per a l's personal. No es pot vendre el *
*  seu contingut sense el previ consentiment per escrit de l'autor.         *
*  El material en aquest fitxer es distribueix "as is" i l'autor no es      *
*  responsabilitza dels danys que pugui causar-ne el seu s.                *
*                                                                           *
*                                                                           *
*  Classe per al control dels processos                                     *
*                                                                           *
************************************************************************** */

// *********************************** INCLUDES
#include "kernel\pcb.h"
#include "errors.h"
#include "kernel\mem.h"
#include "string.h"
#include "kernel\kernel.h"
#include "kernel\asm.h"
#include "kernel\system.h"

#define LDT_ENTRIES			2

extern PCkeKernel _export Kernel;
#ifdef _DEBUG
extern DWORD DebugLevel;
#endif

// **************************************************************************
// Constructor de la classe
// Com a entrada ha de rebre:
// Name    			- Nom del procs
// EntryPoint		- Punt d'entrada al flux inicial del procs
// HeapReserve		- Tamany del 'heap' del procs
// HeapCommit		- 'Bytes' que s'han de comprometre del 'heap' del procs
// StackReserve	- Tamany de la pila del flux inicial del procs
// StackCommit  	- 'Bytes' que s'han de comprometre de la pila del flux inicial
// Parent		  	- Directori pare de l'objecte 'Process'
// PrivilegeLevel	- Nivell de privilegi del procs
// Temp			  	- Procs autodestrut (en aquesta versi no s'utilitza)
// AutoThread   	- Crear automticament el flux inicial
CkeProcess::CkeProcess( PSTR sName, DWORD dwEntryPoint, DWORD dwHeapReserve, DWORD dwHeapCommit, DWORD dwStackReserve, DWORD dwStackCommit, PCkeObject pParent, BOOL bPrivilegeLevel, BOOL bTemp, BOOL bAutoThread ) : CkeResource( pParent, sName, bTemp )
	{
	id	= CID_Process;

	if( bPrivilegeLevel == MODE_KERNEL )
		cbasepriority = 5; // Prioritat estndard per a fluxos del sistema
	else
		cbasepriority = 18; // Prioritat estndard per a fluxos d'usuari

	DWORD flags = CLI();

	bprivilegelevel	= bPrivilegeLevel;
	ppagedir				= new CkePageDirectory( this ); // Preparem el directori de pgines
	PCGDT gdt			= System->GDT();
	WORD index			= gdt->CreateDescriptor();

	{ // Preparem el descriptor de la LDT del procs (dins la GDT)
	PTDescriptor ptdesc = (System->GDT())->Descriptor( index );
	WORD ldtsize = (WORD) (LDT_ENTRIES * sizeof( TDescriptor ));
	DWORD ldtbase = (DWORD) malloc( ldtsize );
	CDescriptor desc( ptdesc );
	desc.Clear();
	desc.SetLDT();
	desc.Base( ldtbase );
	desc.Limit( ldtsize - 1 );
	desc.Granularity( FALSE );
	desc.Present( TRUE );
	}

	pldt = new CLDT( index ); // Preparem la LDT del procs

	{
	// Preparem el descriptor per a codi del procs (dins la LDT)
	WORD index = pldt->CreateDescriptor();
	CDescriptor cdesc( pldt->Descriptor( index ) );
	cdesc.SetCode();
	cdesc.Base( 0 );
	cdesc.Limit( 0xffffffff );
	cdesc.Granularity( TRUE );
	cdesc.Big( TRUE );
	cdesc.SetConformed( FALSE );
	cdesc.SetReadable();

	if( bprivilegelevel )
		cdesc.DPL( 3 ); // Els processos d'usuari tenen DPL = 3
	else
		cdesc.DPL( 0 );

	cdesc.Present( TRUE );

	// Preparem el descriptor per a dades del procs (en la LDT)
	index = pldt->CreateDescriptor();

	cdesc.Descriptor( pldt->Descriptor( index ) );
	cdesc.SetData();
	cdesc.Base( 0 );
	cdesc.Limit( 0xffffffff );
	cdesc.Granularity( TRUE );
	cdesc.Big( TRUE );
	cdesc.SetDown( FALSE );
	cdesc.SetWriteable();

	if( bprivilegelevel )
		cdesc.DPL( 3 ); // Els processos d'usuari tenen DPL = 3
	else
		cdesc.DPL( 0 );

	cdesc.Present( TRUE );
	}

	pthread = NULL; // No hi ha cap flux
	PTHeapItem ptheapitem;

	if( System->Running() ) // Ja hem acabat la inicialitzaci del sistema
		{
		DWORD temp = GetCR3(); // Obtenim el nostre directori de pgines actual
		SetCR3( ppagedir->PDBR() ); // Passem al directori de pgines dest

		// Creem la regi que formar el 'heap' del procs
		PCkeRegion heapregion = new CkeRegion( NULL, dwHeapReserve, REGION_WRITE, NULL );
		// Mapegem la regi
		ptheapitem = (PTHeapItem) ppagedir->MapRegion( heapregion, VA_WRITE );
		// Comprometem un cert tamany de la regi
		ppagedir->Commit( ptheapitem, dwHeapCommit );

		// Inicialitzem el primer element: lliure i ocupa tot el 'heap'
		ptheapitem->Type = HEAP_FREE;
		ptheapitem->dwSize = dwHeapReserve - sizeof( THeapItem );
		ptheapitem->pPrev = ptheapitem->pNext = NULL;

		pheap = new CkeHeap( ptheapitem ); // Creem un nou objecte 'heap'

		SetCR3( temp ); // Tornem al directori de pgines anterior
		}
	else // El procs s el primer -> SYS
		pheap = System->Heap(); // Agafa el 'heap' del sistema

// Crear el primer flux del procs (si ens diuem 'AutoThread')
	if( bAutoThread )
		new CkeThread( dwEntryPoint, dwStackReserve, dwStackCommit, this, NULL );

	STI( flags );
	}

// **************************************************************************
// Destrueix l'objecte
CkeProcess::~CkeProcess()
	{
// Destruir els fluxos que queden -BUG
// Esborrar la LDT i el Heap
	delete pldt;
	delete pheap;
	delete ppagedir; // Esborrem tamb el directori de pgines
	}

// **************************************************************************
// Indica si la classe de l'objecte s o no la que es passa com a parmetre
// Com a entrada ha de rebre:
// Cuid     	- Identificador de la classe
BOOL CkeProcess::ClassCheck( CUID Cuid )
	{
	if( Cuid == CID_Process )
		return TRUE;
	else
		return CkeResource::ClassCheck( Cuid );
	}

// **************************************************************************
// Mapejar una regi de memria. Retorna l'adrea on s'ha mapejat
// Com a entrada ha de rebre:
// Region   	- Regi que volem mapejar
// Protection	- Protecci que volem aplicar a la zona mapejada
PVOID CkeProcess::Map( PCkeRegion pRegion, DWORD cProtection )
	{
	return ppagedir->MapRegion( pRegion, cProtection );
	}

// **************************************************************************
// Desmapeja una regi de memria
// Com a entrada ha de rebre:
// Region   	- Regi que volem desmapejar
VOID CkeProcess::Unmap( PCkeRegion pRegion )
	{
	ppagedir->UnmapRegion( pRegion );
	}

// **************************************************************************
// Retorna cert si hem pogut mapejar la regi en l'adrea especificada
// Com a entrada ha de rebre:
// Region   	- Regi que volem mapejar
// Address  	- Adrea on comenar la regi
// Protection	- Protecci que volem aplicar sobre la memria mapejada
BOOL CkeProcess::TryToMap( PCkeRegion pRegion, PVOID pAddress, DWORD cProtection )
	{
	return ppagedir->TryToMapRegion( pRegion, pAddress, cProtection );
	}

// **************************************************************************
// Compromet memria segons els parmetres
// Com a entrada ha de rebre:
// Address  	- Adrea on comencem a comprometre memria
// Size     	- Tamany de la memria que volem comprometre
VOID CkeProcess::Commit( PVOID pAddress, DWORD dwSize )
	{
	ppagedir->Commit( pAddress, dwSize );
	}

// **************************************************************************
// Afegeix un flux a la llista de fluxos del procs
// Com a entrada ha de rebre:
// NewThread	- Nou flux a afegir
VOID CkeProcess::AddThread( PCkeThread pNewThread )
	{
	DWORD flags = CLI();

	if( pthread )
		pthread->PrevInProc( pNewThread );

	pNewThread->PrevInProc( NULL );
	pNewThread->NextInProc( pthread );
	pthread = pNewThread;

	STI( flags );
	}

// **************************************************************************
// Retorna un apuntador a l'ltim dels fluxos del procs
PCkeThread CkeProcess::Thread()
	{
	return pthread;
	}

// **************************************************************************
// Retorna el valor de la prioritat base dels fluxos del procs
BYTE CkeProcess::BasePriority()
	{
	return cbasepriority;
	}

// **************************************************************************
// Selecciona una nova prioritat base per als fluxos del procs
// Com a entrada ha de rebre:
// Priority 	- Valor de la nova prioritat
VOID CkeProcess::BasePriority( BYTE cPriority )
	{
	DWORD flags = CLI();
	cbasepriority = cPriority;
	STI( flags );
	}

// **************************************************************************
// Retorna l'apuntador al 'heap' del procs
PCkeHeap CkeProcess::Heap()
	{
	return pheap;
	}

// **************************************************************************
// Retorna l'adrea fsica del directori de pgines del procs
DWORD CkeProcess::PDBR()
	{
	return ppagedir->PDBR();
	}

// **************************************************************************
// Retorna el descriptor que fa referncia a la LDT del procs
WORD CkeProcess::LDTEntry()
	{
	return pldt->GDT();
	}

// **************************************************************************
// Obt el nivell de privilegi del procs (Nucli / Usuari)
BOOL CkeProcess::PrivilegeLevel()
	{
	return bprivilegelevel;
	}

