/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel keyboard code.                 ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  03. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <kernel/selector.h>

#include <kernel/kernel.h>
#include <kernel/irq.h>
#include <kernel/keyboard.h>
#include <kernel/timer.h>
#include <kernel/schedule.h>
#include <kernel/waitq.h>
#include <kernel/kprint.h>
#include <kernel/console.h>

// undef this to work silence...
#undef KBD_DEBUG

/*
 * Simple Keyboard Service
 *
 * TODO:
 *	Dead keys
 *	API, loading diff. Key tables (zumindest amerik. Tastatur!)
 *	Konstanten als Defines
 *	Ext. Keys (Cursortasten und so)
 *	Timeouts ueber timer (wenn einer da ist)
 */

#define countof(x) (sizeof(x) / sizeof(*x))

// Forward declarations:
static void KeyboardInterrupt (IntRegs *, long);

/*
 * Scancode to ascii table (German Keyboard)
 */

// unshifted Keys:
static unsigned char NormalKeys [] =
{
  0x00, 0x1b,  '1',  '2',  '3',  '4',  '5',  '6',  '7',  '8',  '9',  '0', 0xe1, 0x27, '\b', '\t',
   'q',  'w',  'e',  'r',  't',  'z',  'u',  'i',  'o',  'p', 0x81, 0x2b, 0x0c, 0x00,  'a',  's',
   'd',  'f',  'g',  'h',  'j',  'k',  'l', 0x94, 0x84, 0x5e, 0x00, 0x23,  'y',  'x',  'c',  'v',
   'b',  'n',  'm',  ',',  '.',  '-', 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c
};

// Shifted Keys:
static unsigned char ShiftedKeys [] =
{
  0x00, 0x1b,  '!', 0x22, 0x15,  '$',  '%',  '&',  '/',  '(',  ')',  '=',  '?', 0x60, '\b', '\t',
   'Q',  'W',  'E',  'R',  'T',  'Z',  'U',  'I',  'O',  'P', 0x9a, 0x2a, 0x0c, 0x00,  'A',  'S',
   'D',  'F',  'G',  'H',  'J',  'K',  'L', 0x99, 0x8e, 0xf8, 0x00, 0x27,  'Y',  'X',  'C',  'V',
   'B',  'N',  'M',  ';',  ':',  '_', 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e
};


/*
 * State of the control keys (ctrl, alt, shift, ...)
 */
static byte ControlKeys;

/*
 * State of keyboard and LED's
 */
static WaitQ *LEDwaitq = 0;            // waitqueue for keyboard status changes

static byte Lastledstate;              // last keyboard LED state

static byte NumLock;                   // Num lock is on
static byte CapsLock;                  // Caps lock is on
static byte ScrollLock;                // Scroll lock is on

/*
 * Key Queue
 *
 */
#define KEYBOARD_BUFSIZE   32
typedef struct Key_queue
{
	unsigned long head;
	unsigned long tail;
	unsigned char Keys[KEYBOARD_BUFSIZE];
} Key_queue;

static WaitQ *Keywaitq = 0;
static Key_queue KeyQueue;

/*
 * Assigned HotKeys
 */
struct HotKey
{
   ushort   ScanCode;                  // SCANCODE of Hotkey
   byte     Modi;                      // flags for alt/shift/ctrl
   void   (*fnc)(void);                // function to call on hotkey
};

static struct HotKey HotKeys [16];     // current assigned hotkeys
static ushort CountOfHotKeys = 0;      // # of assigned hotkeys

struct DebuggerHotKey
{
	ushort	ScanCode;						// SCANCODE of Hotkey
	byte	Modi;							// flags for alt/shift/ctrl
	void	(*fnc)(IntRegs *, int);			// function to call on hotkey
};

static struct DebuggerHotKey DebugHotkey;      // Hotkey only for Debug usage

/*
 * Wait for Keyboard to become ready
 */
static void Kbd_Wait (void)
{
    int i;

    for (i = 0; i < 0x10000; i++)
    	 if ((inportb_p(0x64) & 0x02) == 0)
		    return;

    kprint ("Kernel: Keyboard timed out!!\r\n");
}

/*
 * Send a command to keyboard
 */
static void Kbd_SendCmd (byte cmd)
{
    Kbd_Wait ();
    outportb_p (0x64, cmd);
}

/*
 * Send data to keyboard
 */
static void Kbd_SendData (byte data)
{
	Kbd_Wait();
	outportb_p (0x60, data);
   Kbd_Wait ();
}

/*
 * Set the Keyboard LED's
 *
 * This is an independend thread.
 */
static void SetKeyboardLEDs (void)
{
   for(;;)
   {
      // build new LED mask
      byte newledstate = 0;

      newledstate |= (ScrollLock) ? 1 : 0;
      newledstate |= (NumLock)    ? 2 : 0;
      newledstate |= (CapsLock)   ? 4 : 0;

      // has the mask changed then update the LED's
      if (Lastledstate != newledstate)
      {
         Lastledstate = newledstate;

	      Kbd_SendData(0xed);
         Kbd_SendData(newledstate);
      }

      // and wait for the next change
      SleepOn (&LEDwaitq);
   }
}

extern void *Lin2Ptr(ulong lin);

static void CallDebugger (IntRegs *regs, void (*fnc)(IntRegs *, int))
{
   int    i;

   // the keyboard is disabled, so enable Keyboard
	Kbd_SendCmd (0xae);

	// and ACK last read
	i = inportb_p (0x61);
	outportb_p (0x61, i | 0x80);
	outportb_p (0x61, i);

   fnc (regs, -1);
}

/*
 * Check if the current key is a hotkey
 */
static int CheckForHotKey (IntRegs *regs, ushort ScanCode)
{
   int i;

   if (DebugHotkey.fnc != 0 && DebugHotkey.ScanCode == ScanCode && ((DebugHotkey.Modi & ControlKeys) == DebugHotkey.Modi))
   {
      CallDebugger (regs, DebugHotkey.fnc);
      return (1);
   }


   // no hotkey assigned, nothing to do...
   if (CountOfHotKeys == 0)
      return (0);

   for (i = 0; i < countof(HotKeys); i++)
   {
       if (HotKeys[i].fnc != 0 && HotKeys[i].ScanCode == ScanCode && ((HotKeys[i].Modi & ControlKeys) == HotKeys[i].Modi))
       {
            HotKeys[i].fnc();
            return (1);
       }
   }

   return (0);
}

/*
 *
 */
static void QueueNewKey (char Key)
{
   WakeUp (&Keywaitq);

   if ((KeyQueue.tail - KeyQueue.head -1) & (KEYBOARD_BUFSIZE -1))
   {
      KeyQueue.Keys[KeyQueue.head] = Key;
      KeyQueue.head = (KeyQueue.head +1) & (KEYBOARD_BUFSIZE -1);
   }
}

static int TranslateScancode (IntRegs *regs, int code, int InDebugMode)
{
   static int lastcode = 0;
   byte LedChanged = 0;

#ifdef KBD_DEBUG
	char			buf[9];
#endif

	if (code == 0xe0 || code == 0xe1)
	{
	    // remeber prefix-code
	    lastcode = code;
	}
	else
	{
#ifdef KBD_DEBUG
	    GotoXY(0,9);
	    L2Str(buf, lastcode);
	    buf[8] = '\0';
	    kprint("last: ");
	    kprint(buf);
	    L2Str(buf, code);
	    buf[8] = '\0';
	    kprint(", code: ");
	    kprint(buf);
#endif

	    code |= (lastcode << 8);
	    switch(code)
	    {
		case 0x2a:
         ControlKeys |= (KBD_SHIFT | KBD_LEFTSHIFT);
			code = 0;
			break;

		case 0x36:
         ControlKeys |= (KBD_SHIFT | KBD_RIGHTSHIFT);
			code = 0;
			break;

		case 0xaa:
         ControlKeys &= ~KBD_LEFTSHIFT;
         if (! (ControlKeys & KBD_RIGHTSHIFT))
            ControlKeys &= ~KBD_SHIFT;
			code = 0;
         break;

		case 0xb6:
         ControlKeys &= ~KBD_RIGHTSHIFT;
         if (! (ControlKeys & KBD_LEFTSHIFT))
            ControlKeys &= ~KBD_SHIFT;
			code = 0;
			break;

		case 0x1d:
         ControlKeys |= (KBD_CTRL | KBD_LEFTCTRL);
			code = 0;
			break;

      case 0xe01d:
         ControlKeys |= (KBD_CTRL | KBD_RIGHTCTRL);
			code = 0;
			break;

		case 0x9d:
         ControlKeys &= ~KBD_LEFTCTRL;
         if (! (ControlKeys & KBD_RIGHTCTRL))
            ControlKeys &= ~KBD_CTRL;
			code = 0;
			break;

		case 0xe09d:
         ControlKeys &= ~KBD_RIGHTCTRL;
         if (! (ControlKeys & KBD_LEFTCTRL))
            ControlKeys &= ~KBD_CTRL;
			code = 0;
			break;

		case 0x38:
         ControlKeys |= KBD_ALT;
			code = 0;
			break;
		case 0xb8:
         ControlKeys &= ~KBD_ALT;
			code = 0;
			break;

		case 0xe038:
         ControlKeys |= KBD_ALTGR;
			code = 0;
			break;

		case 0xe0b8:
         ControlKeys &= ~KBD_ALTGR;
			code = 0;
			break;

		case 0x53:
		case 0xe053:
			if (ControlKeys & (KBD_ALT | KBD_CTRL))
				Reboot (0);
			code = 0;
			break;

      case 0x0045:
           NumLock = (NumLock) ? 0 : 1;
           LedChanged = 1;
           break;

      case 0x003a:
           CapsLock = (CapsLock) ? 0 : 1;
           LedChanged = 1;
           break;

      case 0x0046:
           ScrollLock = (ScrollLock) ? 0 : 1;
           LedChanged = 1;
           break;
	    }

       if (LedChanged && !InDebugMode)
          WakeUp (&LEDwaitq);

#ifdef KBD_DEBUG
	    kprint ((ControlKeys & KBD_SHIFT      ) ? "S" : "s");
	    kprint ((ControlKeys & KBD_LEFTSHIFT  ) ? "S" : "s");
	    kprint ((ControlKeys & KBD_RIGHTSHIFT ) ? "S" : "s");
	    kprint ((ControlKeys & KBD_CTRL       ) ? "C" : "c");
	    kprint ((ControlKeys & KBD_LEFTCTRL   ) ? "C" : "c");
	    kprint ((ControlKeys & KBD_RIGHTCTRL  ) ? "C" : "c");
	    kprint ((ControlKeys & KBD_ALT        ) ? "A" : "a");
	    kprint ((ControlKeys & KBD_ALTGR      ) ? "G" : "g");
#endif

       if (InDebugMode || ! CheckForHotKey (regs, code))
       {
          if (code >= 0 && code < (sizeof (NormalKeys) / sizeof(*NormalKeys)))
	       {
		       code = (ControlKeys & KBD_SHIFT) ? ShiftedKeys[code] : NormalKeys[code];

		       if (code)
		       {
#ifdef KBD_DEBUG
		          buf[0] = ' ';
		          buf[1] = '\'';
		          buf[2] = code;
		          buf[3] = '\'';
		          buf[4] = 0;
		          kprint(buf);
		          kprint(" ");
                B2Str(buf, code);
                buf[2] = 0;
	             kprint(buf);
#endif
                if (code >= 0x60 && (ControlKeys & KBD_CTRL))
                   code -= 0x5f;
		       }
          }
          else
             code = 0;
       }
       else
           code = 0;

	    lastcode = 0;
	}
   return (code);
}

/*
 * Keyboard Interrupt
 *
 */
static void KeyboardInterrupt (IntRegs *regs, long NotUsed)
{
   int code, status;
   byte i;

   NotUsed = NotUsed;

	// Disable Keyboard
	Kbd_SendCmd (0xad);

	// Wait for keyboard
	Kbd_Wait ();

   // Get Keyboard Status
	status = inportb_p (0x64);

   // Scancode avail?
   if ((status & 1) == 1)
   {
   	// Get Scancode
	   code = inportb_p(0x60);

      code = TranslateScancode (regs, code, 0);

      if (code)
         QueueNewKey (code);
   }

	// Enable Keyboard
	Kbd_SendCmd (0xae);

	// and ACK last read
	i = inportb_p (0x61);
	outportb_p (0x61, i | 0x80);
	outportb_p (0x61, i);
}

////////////////////////////////////////////////////////////////////////
// Public Functions:

/*
 * Setup Keyboard Interrupt
 */
void InitKeyboard ()
{
   byte i;

	// Clear Keyboard queue
	inportb_p(0x60);
	Kbd_Wait ();

   ControlKeys = 0;

   Lastledstate = 0xff;

   NumLock     = 0;     // numlock    off
   CapsLock    = 0;     // capslock   off
   ScrollLock  = 0;     // scrolllock off

   KeyQueue.tail = KeyQueue.head = 0;
   Keywaitq = 0;

   CountOfHotKeys = 0;

   for (i = 0; i < countof(HotKeys); i++)
   {
       HotKeys[i].ScanCode = 0;
       HotKeys[i].fnc = 0;
   }

	// Set Interrupt procedure
   RequestIRQ (1, KeyboardInterrupt, 0);

	// Enable Keyboard
	Kbd_SendCmd (0xae);

	// and ACK Last Read
	i = inportb_p (0x61);
	outportb_p (0x61, i | 0x80);
	outportb_p (0x61, i);

   CreateKernelTask   (SetKeyboardLEDs, 0);
}

/*
 * read the next key, waits if there is none!
 */
int getch (void)
{
   int Key;

   if (KeyQueue.tail == KeyQueue.head)
      SleepOn (&Keywaitq);

   Key = (int) KeyQueue.Keys[KeyQueue.tail];
   KeyQueue.tail = (KeyQueue.tail +1) & (KEYBOARD_BUFSIZE -1);

   return (Key);
}

/*
 * read the next Key (returns -1 if there is none)
 */
int GetKey (void)
{
   if (KeyQueue.tail == KeyQueue.head)
      return (-1);

   return (getch());
}

/*
 * Reboot the machine
 */
void Reboot (int DoColdBoot)
{
   if (DoColdBoot)
	   PokeW(SEL_DLINEAR, 0x0472, 0x0000);
   else
      // Disable Memory test
	   PokeW(SEL_DLINEAR, 0x0472, 0x1234);

   // Send Reboot to the kayboard controller
   Kbd_SendCmd (0xfe);
}

int SetHotkey (int Scancode, byte Modi, void (*fnc)(void))
{
   int i;

   for (i = 0; i < countof(HotKeys); i++)
   {
      if (fnc == 0)
      {
         if (HotKeys[i].ScanCode == Scancode && HotKeys[i].ScanCode == Modi)
         {
            HotKeys[i].ScanCode = 0;
            HotKeys[i].fnc = 0;

            CountOfHotKeys --;
            return (1);
         }
      }
      else
      {
         if (HotKeys[i].fnc == 0)
         {
            HotKeys[i].ScanCode = Scancode;
            HotKeys[i].Modi = Modi;
            HotKeys[i].fnc  = fnc;

            CountOfHotKeys ++;
            return (1);
         }
      }
   }
   return (0);
}

void SetDebuggerHotkey (int Scancode, byte Modi, void (*fnc)(IntRegs *, int))
{
   DebugHotkey.ScanCode = Scancode;
   DebugHotkey.Modi     = Modi;
   DebugHotkey.fnc      = fnc;
}

int GetKeyForDebugger (int WaitForKey)
{
   int code, status;
   byte i;

   for (;;)
   {
      // Get Keyboard Status
	   status = inportb_p (0x64);

      // Scancode avail?
      if ((status & 1) == 1)
      {
      	// Disable Keyboard
	      Kbd_SendCmd (0xad);

	      // Wait for keyboard
	      Kbd_Wait ();

      	// Get Scancode
	      code = inportb_p(0x60);

         code = TranslateScancode (0, code, 1);

	      // Enable Keyboard
	      Kbd_SendCmd (0xae);

	      // and ACK last read
	      i = inportb_p (0x61);
	      outportb_p (0x61, i | 0x80);
	      outportb_p (0x61, i);

         if (code != 0)
            return (code);
      }
      else if (! WaitForKey)
         return (-1);
   }

   // not reached
   return(0);
}

