/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel code.                          ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  03. apr 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <string.h>
#include <kernel/kernel.h>
#include <kernel/kprint.h>

typedef struct MSDOS_BootSector
{
	 byte    bs_ignored[3]; 		// Boot strap short or near jump
	 byte    bs_system_id[8]; 		// disk id
	 ushort  bs_sector_size; 		// bytes per logical sector
	 byte    bs_cluster_size; 		// sectors/cluster
	 ushort  bs_reserved; 			// reserved sectors
	 byte    bs_fats; 				// number of FATs 1..2
	 ushort  bs_dir_entries; 		// root directory entries
	 ushort  bs_sectors; 			// number of sectors
	 byte    bs_media; 				// media code
	 ushort  bs_fat_length; 		// sectors/FAT
	 ushort  bs_secs_track; 		// sectors per track
	 ushort  bs_heads; 				// number of heads
	 ulong   bs_hidden; 				// hidden sectors (unused)
	 ulong   bs_total_sect; 		// number of sectors (if sectors == 0)
	 byte    padding[512 - 36];
} MSDOS_BootSector;

typedef struct MSDOS_DirEntry
{

	 byte    name[8];
	 byte    ext[3];
	 byte    attrib;
	 byte    reserved[10];
	 ushort  time;
	 ushort  date;
	 ushort  start_cluster;
	 ulong   file_size;
} MSDOS_DirEntry;

#define FILEOPEN	  1
#define FILEREAD	  2
#define FILEWRITE	  4
#define FILEAPPEND  8
#define FILEINUSE  16

typedef struct FileStruct
{
	char 				Name [16];
	ulong    		offset;
	byte				Mode;
	MSDOS_DirEntry de;
} FileStruct;

static int IsInited = 0;

static MSDOS_BootSector BootSector;

#define CACHEBUFFERS	32
static char					Cache      [CACHEBUFFERS][512];
static ulong				CacheSector[CACHEBUFFERS];
static char					CacheDirty [CACHEBUFFERS];
static ulong				CacheASL   [CACHEBUFFERS];;

static long softreads  = 0;
static long hardreads  = 0;
static long softwrites = 0;
static long hardwrites = 0;

static ushort FatStart;
static ushort DirStart;
static ushort DataStart;
static char   Fat16;

#define NROFFILES	16
FileStruct files[NROFFILES];

extern int RamDiskReadBlock  (int Nr, char *Buffer);
extern int RamDiskWriteBlock (int Nr, char *Buffer);


static int DeviceRead (ulong Nr, void *Buffer)
{
	hardreads ++;

	RamDiskReadBlock (Nr, (char *)Buffer);
//	absread (0, 1, Nr, Buffer);

	return (0);
}

static int DeviceWrite (ulong Nr, void *Buffer)
{
	hardwrites ++;

	RamDiskWriteBlock (Nr, (char *)Buffer);
//	abswrite (0, 1, Nr, Buffer);

	return (0);
}

static int FindInCache (ulong Nr)
{
	int i;
	int found = -1;

	for (i = 0; i < CACHEBUFFERS; i++)
	{
		if (CacheSector[i] == Nr)
		{
			CacheASL[i]++;
			found = i;
			break;
		}
	}

	return (found);
}

static int FindFreeCache (void)
{
	int i, oldest = 0;
	ulong old = 0XFFFFFFFF;

	for (i = 0; i < CACHEBUFFERS; i++)
	{
		if (CacheSector[i] == -1)
		{
			return (i);
		}
		else if (CacheASL[i] < old)
		{
			old = CacheASL[i];
			oldest = i;
		}
	}
	if (CacheDirty[oldest])
	{
		DeviceWrite (CacheSector[oldest], Cache[oldest]);
		CacheDirty[oldest] = 0;
	}
	return (oldest);
}

int ReadSector (ulong Nr, void *Buffer)
{
	int i, found = -1;

	softreads ++;

	found = FindInCache (Nr);

	if (found == -1)
	{
		found = FindFreeCache ();
		DeviceRead (Nr, Cache[found]);
		CacheSector[found] = Nr;
		CacheDirty[found] = 0;
		CacheASL[found] = 0;

		if (Nr >= FatStart && Nr < DirStart)
			CacheASL[found] += 10;
	}

	for (i = 0; i < 512; i++)
		((char *) Buffer)[i] = Cache[found][i];

	return (1);
}

int WriteSector (ulong Nr, void *Buffer)
{
	int i, found = -1;

	softwrites ++;

	found = FindInCache (Nr);

	if (found == -1)
	{
		found = FindFreeCache ();
		CacheSector[found] = Nr;
		CacheDirty[found] = 0;
		CacheASL[found] = 0;
	}

	for (i = 0; i < 512; i++)
		Cache[found][i] = ((char *) Buffer)[i];

	CacheDirty[found] = 1;

	return (1);
}

void FlushCache(void)
{
	int i;

	if (IsInited != 1)
		return;

	for (i = 0; i < CACHEBUFFERS; i++)
	{
		if (CacheSector[i] != -1 && CacheDirty[i])
		{
			DeviceWrite (CacheSector[i], Cache[i]);
			CacheDirty[i] = 0;
		}
	}
}

int InitMiniFS (void)
{
	int i;
	IsInited = 0;

	for (i = 0; i < CACHEBUFFERS; i++)
		CacheSector[i] = -1;

	for (i = 0; i < NROFFILES; i++)
		files[i].Mode = 0;

	// try to read the bootsektor
	if (ReadSector (0, &BootSector) < 0)
	{
		kprintf ("can't read bootsector of drive A!\n");
		return (0);
	}
	if (BootSector.bs_cluster_size != 1)
	{
		kprintf ("can only handle 1 sector / cluster!\n");
		return (0);
	}
	if (BootSector.bs_sector_size != 512)
	{
		kprintf ("can only handle 512 bytes sectors!\n");
		return (0);
	}
	if (BootSector.bs_media == 0xF0)
		Fat16 = 0;
	else
		Fat16 = 1;

	FatStart  = BootSector.bs_reserved;
	DirStart  = FatStart + BootSector.bs_fats * BootSector.bs_fat_length;
	DataStart = DirStart + (BootSector.bs_dir_entries*32+31)/512;

	IsInited = 1;

	return (1);
}

int FindFile (const char *name, MSDOS_DirEntry *dire)
{
	char            Buffer[512];
	char				 Temp[12], *c;
	MSDOS_DirEntry *de;
	int             i, j;

	if (! name || !*name)
		return (0);

	for (i = 0; i < 12; i++)
		Temp[i] = ' ';
	c = strchr (name, '.');
	if (c)
		*c++ = 0;
	for (i = 0; i < 8; i++)
	{
		if (! name[i])
			break;
		Temp[i] = name[i];
	}
	if (c)
	{
		for (i = 0; i < 3; i++)
		{
			if (! c[i])
				break;
			Temp[8 + i] = c[i];
		}
	}
	Temp[11] = '\0';
	strlwr(Temp);

	for (i = 0; i < ((BootSector.bs_dir_entries*32+31)/512); i++)
	{
		ReadSector (DirStart +i, Buffer);

		for (j = 0, de = (MSDOS_DirEntry *) Buffer; j < (512 / 32); j++, de++)
		{
			if (de -> name[0] == 0x00 || de -> name[0] == 0xE5)
				continue;

			if (! strncmp (de -> name, Temp, 11))
			{
				*dire = *de;
				return (1);
			}
		}
	}
	return (0);
}

ushort FollowClusterChain (ushort startcluster)
{
	byte   Buffer[512];
	int    FatSektor;
	int    FatOffset;
	byte   FatByte1, FatByte2;
	ushort nextcluster;

	if (Fat16)
	{
		FatSektor = (startcluster * 16/8) / 512;
		FatOffset = (startcluster * 16/8) % 512;
	}
	else
	{
		FatSektor = (startcluster * 12/8) / 512;
		FatOffset = (startcluster * 12/8) % 512;
	}

	ReadSector (FatStart + FatSektor, Buffer);
	FatByte1 = Buffer[FatOffset];
	if (FatOffset == 511)
	{
		ReadSector (FatStart + FatSektor +1, Buffer);
		FatByte2 = Buffer[0];
	}
	else
		FatByte2 = Buffer[FatOffset +1];

	if (Fat16)
	{
		nextcluster = (FatByte1 & 0xff) | ((FatByte2 & 0xFf) << 8);
		if((nextcluster & 0xfff8) == 0xfff8)
			nextcluster = 0;

	}
	else
	{
		if (startcluster & 1)
			nextcluster = (FatByte2 << 4) | ((FatByte1 >> 4) & 0x0f);
		else
			nextcluster = (FatByte1 & 0xff) | ((FatByte2 & 0x0f) << 8);
		if((nextcluster & 0x0ff8) == 0x0ff8)
			nextcluster = 0;
	}

	return (nextcluster);
}

int fileopen (char *name, int mode)
{
	int i;
	int fd = -1;

	for (i = 0; i < NROFFILES; i++)
	{
		if (files[i].Mode == 0)
		{
			files[i].Mode = FILEINUSE;
			fd = i;
			break;
		}
	}

	if (fd == -1)
		return (-1);

	if (! FindFile (name, & files[fd].de))
	{
		files[fd].Mode = 0;
		return (-1);
	}

	files[fd].Name [0] = 0;

	if ((mode & (FILEWRITE | FILEAPPEND)) == (FILEWRITE | FILEAPPEND))
		files[fd].offset   = files[fd].de.file_size;
	else
		files[fd].offset   = 0;

	files[fd].Mode     = mode;

	return (fd);
}

#define min(a,b) (((a) < (b)) ? a : b)

int fileread (int fd, char *buffer, int len)
{
	char Buffer[512];
	int BlockNo;
	int Offset;
	int i, j;
	int readed = 0;
	ushort Cluster;
	buffer = buffer;
	len    = len;

	if (! (files[fd].Mode & FILEREAD))
		return (-1);

	if (files[fd].offset >= files[fd].de.file_size)
		return (0);

	len = min (len, (files[fd].de.file_size - files[fd].offset));

	if (! buffer || len <= 0)
		return (0);

	BlockNo = files[fd].offset / 512;
	Offset  = files[fd].offset % 512;

	Cluster = files[fd].de.start_cluster;
	for (i = 0; i < BlockNo; i++)
	{
		Cluster = FollowClusterChain (Cluster);
		if (Cluster <= 0)
			return (-1);
	}

	ReadSector (Cluster, Buffer);
	i = 512 - Offset;
	i = min(i, len);

	// erster Teile
	for (j = 0; j < i; j++)
		buffer[j] = Buffer[Offset + j];
	len -= i;
	readed += i;
	files[fd].offset += i;

	if (len <= 0)
		return (readed);

	// alle Sektoren
	for(;;)
	{
		Cluster = FollowClusterChain (Cluster);
		if (Cluster <= 0)
			return (readed);
		ReadSector (Cluster, Buffer);
		i = 512;
		i = min(i, len);
		for (j = 0; j < i; j++)
			buffer[j] = Buffer[Offset + j];
		len -= i;
		readed += i;
		files[fd].offset += i;

		if (len <= 0)
			return (readed);
	}
}

int filewrite (int fd, char *buffer, int len)
{
	buffer = buffer;
	len    = len;

	if (! (files[fd].Mode & FILEWRITE))
		return (-1);

	return (-1);
}

int fileseek (int fd, ulong offset, int whence)
{
	fd     = fd;
	offset = offset;
	whence = whence;

	return (-1);
}

int fileclose (int fd)
{
	files[fd].Mode = 0;

	return (0);
}

/*
void main ()
{
	ushort u;
	int fd;
	char buffer[2048];

	if (! InitMiniFS ())
	{
		kprintf ("can't init FS!\n");
		return;
	}
	else
		kprintf ("FS initialized!\n");

	fd = fileopen ("test.dat", FILEREAD);

	if (fd < 0)
		kprintf ("File not found\n");
	else
	{
		while (fileread (fd, buffer, sizeof(buffer)) > 0)
			;
		fileclose (fd);
	}

	kprintf ("Reads  %ld/%ld, Writes %ld/%ld\n", softreads, hardreads, softwrites, hardwrites);

	FlushCache ();
}
*/
