/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel DMA Handling code.             ***/
/***    portions Copyright (C) 1997 by Martin Kortmann                    ***/
/***                                                                      ***/
/*** based on the linux dma code written by Hennus Bergman, 1992          ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  05. apr 1997  Coding started (MK)                                   ***/
/****************************************************************************/

#ifndef __DMA_H
#define __DMA_H

#include <kernel/kernel.h>

#define MAX_DMA_CHANNELS	8
#define MAX_DMA_ADDRESS    0x1000000

/* 8237 DMA controllers */
#define IO_DMA1_BASE			0x00	/* 8 bit slave DMA, channels 0..3 */
#define IO_DMA2_BASE			0xC0	/* 16 bit master DMA, ch 4(=slave input)..7 */

/* DMA controller registers */
#define DMA1_CMD_REG			0x08	/* command register (w) */
#define DMA1_STAT_REG		0x08	/* status register (r) */
#define DMA1_REQ_REG    	0x09    /* request register (w) */
#define DMA1_MASK_REG		0x0A	/* single-channel mask (w) */
#define DMA1_MODE_REG		0x0B	/* mode register (w) */
#define DMA1_CLEAR_FF_REG	0x0C	/* clear pointer flip-flop (w) */
#define DMA1_TEMP_REG      0x0D    /* Temporary Register (r) */
#define DMA1_RESET_REG		0x0D	/* Master Clear (w) */
#define DMA1_CLR_MASK_REG  0x0E    /* Clear Mask */
#define DMA1_MASK_ALL_REG  0x0F    /* all-channels mask (w) */

#define DMA2_CMD_REG			0xD0	/* command register (w) */
#define DMA2_STAT_REG		0xD0	/* status register (r) */
#define DMA2_REQ_REG       0xD2    /* request register (w) */
#define DMA2_MASK_REG		0xD4	/* single-channel mask (w) */
#define DMA2_MODE_REG		0xD6	/* mode register (w) */
#define DMA2_CLEAR_FF_REG	0xD8	/* clear pointer flip-flop (w) */
#define DMA2_TEMP_REG      0xDA    /* Temporary Register (r) */
#define DMA2_RESET_REG		0xDA	/* Master Clear (w) */
#define DMA2_CLR_MASK_REG  0xDC    /* Clear Mask */
#define DMA2_MASK_ALL_REG  0xDE    /* all-channels mask (w) */

#define DMA_ADDR_0         0x00    /* DMA address registers */
#define DMA_ADDR_1         0x02
#define DMA_ADDR_2         0x04
#define DMA_ADDR_3         0x06
#define DMA_ADDR_4         0xC0
#define DMA_ADDR_5         0xC4
#define DMA_ADDR_6         0xC8
#define DMA_ADDR_7         0xCC

#define DMA_CNT_0          0x01    /* DMA count registers */
#define DMA_CNT_1          0x03
#define DMA_CNT_2          0x05
#define DMA_CNT_3          0x07
#define DMA_CNT_4          0xC2
#define DMA_CNT_5          0xC6
#define DMA_CNT_6          0xCA
#define DMA_CNT_7          0xCE

#define DMA_PAGE_0         0x87    /* DMA page registers */
#define DMA_PAGE_1         0x83
#define DMA_PAGE_2         0x81
#define DMA_PAGE_3         0x82
#define DMA_PAGE_5         0x8B
#define DMA_PAGE_6         0x89
#define DMA_PAGE_7         0x8A

#define DMA_MODE_READ		0x44	/* I/O to memory, no autoinit, increment, single mode */
#define DMA_MODE_WRITE		0x48	/* memory to I/O, no autoinit, increment, single mode */
#define DMA_MODE_CASCADE 	0xC0   /* pass thru DREQ->HRQ, DACK<-HLDA only */

/* enable/disable a specific DMA channel */
static __inline__ void enable_dma(unsigned int dmanr)
{
	if (dmanr<=3)
		outportb_p(DMA1_MASK_REG, dmanr);
	else
		outportb_p(DMA2_MASK_REG, dmanr & 3);
}

static __inline__ void disable_dma(unsigned int dmanr)
{
	if (dmanr<=3)
		outportb_p(DMA1_MASK_REG, dmanr | 4);
	else
		outportb_p(DMA2_MASK_REG, (dmanr & 3) | 4);
}

static __inline__ void clear_dma_ff(unsigned int dmanr)
{
	if (dmanr<=3)
		outportb_p(DMA1_CLEAR_FF_REG, 0);
	else
		outportb_p(DMA2_CLEAR_FF_REG, 0);
}

/* set mode (above) for a specific DMA channel */
static __inline__ void set_dma_mode(unsigned int dmanr, char mode)
{
	if (dmanr<=3)
		outportb_p(DMA1_MODE_REG, mode | dmanr);
	else
		outportb_p(DMA2_MODE_REG, mode | (dmanr&3));
}

/* Set only the page register bits of the transfer address.
 * This is used for successive transfers when we know the contents of
 * the lower 16 bits of the DMA current address register, but a 64k boundary
 * may have been crossed.
 */
static __inline__ void set_dma_page(unsigned int dmanr, char pagenr)
{
	switch(dmanr)
	{
		case 0:
			outportb_p(DMA_PAGE_0, pagenr);
			break;
		case 1:
			outportb_p(DMA_PAGE_1, pagenr);
			break;
		case 2:
			outportb_p(DMA_PAGE_2, pagenr);
			break;
		case 3:
			outportb_p(DMA_PAGE_3, pagenr);
			break;
		case 5:
			outportb_p(DMA_PAGE_5, pagenr & 0xfe);
			break;
		case 6:
			outportb_p(DMA_PAGE_6, pagenr & 0xfe);
			break;
		case 7:
			outportb_p(DMA_PAGE_7, pagenr & 0xfe);
			break;
	}
}


/* Set transfer address & page bits for specific DMA channel.
 * Assumes dma flipflop is clear.
 */
static __inline__ void set_dma_addr(unsigned int dmanr, unsigned int a)
{
	set_dma_page(dmanr, a>>16);
	if (dmanr <= 3)
	{
	    outportb_p(((dmanr&3)<<1) + IO_DMA1_BASE, a & 0xff);
       outportb_p(((dmanr&3)<<1) + IO_DMA1_BASE, (a>>8) & 0xff);
	}
	else
	{
	    outportb_p(((dmanr&3)<<2) + IO_DMA2_BASE, (a>>1) & 0xff);
	    outportb_p(((dmanr&3)<<2) + IO_DMA2_BASE, (a>>9) & 0xff);
	}
}


/* Set transfer size (max 64k for DMA1..3, 128k for DMA5..7) for
 * a specific DMA channel.
 * You must ensure the parameters are valid.
 * NOTE: from a manual: "the number of transfers is one more
 * than the initial word count"! This is taken into account.
 * Assumes dma flip-flop is clear.
 * NOTE 2: "count" represents _bytes_ and must be even for channels 5-7.
 */
static __inline__ void set_dma_count(unsigned int dmanr, unsigned int count)
{
   count--;
	if (dmanr <= 3)
	{
	    outportb_p(((dmanr&3)<<1) + 1 + IO_DMA1_BASE, count & 0xff);
	    outportb_p(((dmanr&3)<<1) + 1 + IO_DMA1_BASE, (count>>8) & 0xff);
   }
	else
	{
	    outportb_p(((dmanr&3)<<2) + 2 + IO_DMA2_BASE, (count>>1) & 0xff);
	    outportb_p(((dmanr&3)<<2) + 2 + IO_DMA2_BASE, (count>>9) & 0xff);
   }
}


/* Get DMA residue count. After a DMA transfer, this
 * should return zero. Reading this while a DMA transfer is
 * still in progress will return unpredictable results.
 * If called before the channel has been used, it may return 1.
 * Otherwise, it returns the number of _bytes_ left to transfer.
 *
 * Assumes DMA flip-flop is clear.
 */
static __inline__ int get_dma_residue(unsigned int dmanr)
{
	unsigned int io_port = (dmanr<=3)? ((dmanr&3)<<1) + 1 + IO_DMA1_BASE
					 : ((dmanr&3)<<2) + 2 + IO_DMA2_BASE;

	/* using short to get 16-bit wrap around */
	unsigned short count;

	count = 1 + inportb_p(io_port);
	count += inportb_p(io_port) << 8;
	
	return (dmanr<=3)? count : (count<<1);
}


/* These are in kernel/dma.c: */
extern int RequestDMA(unsigned int dmanr, const char * device_id);	/* reserve a DMA channel */
extern void FreeDMA  (unsigned int dmanr);	/* release it again */


#endif // __DMA_H
