/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel header file.                   ***/
/*** It includes commonly used typedefs and function prototypes           ***/
/***    Copyright (C) 1997 by Joachim Breitsprecher                       ***/
/***       email: j.breitsprecher@schwaben.de                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here)  ***/
/***  Joachim Breitsprecher (HJB)                                         ***/
/***      <j.breitsprecher@schwaben.de>                                   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  22. Feb 1997  Coding started (HJB)                                  ***/
/****************************************************************************/
// Some useful typedefs
#ifndef __KERNEL_H
#define __KERNEL_H

typedef unsigned char	byte;
typedef unsigned short	ushort;
typedef unsigned int	uint;
typedef unsigned long	ulong;

typedef enum
{
	FALSE = 0,
	TRUE
} boolean;

// The structure of a segment descriptor
struct DESCRIPTOR
{
	ushort		limit0;			// Segment limit, bits 0-15
	ushort		base0;			// Segment base, bits 0-15
	byte		base1;			// Segment base, bits 16-23
	byte		type;			// Segment type, sys flag, dpl, present flag
	byte		limit1;			// Segment limit, bits 16-19, opsize, gran.
	byte		base2;			// Segment base, bits 24-31
} __attribute__((packed));

typedef struct DESCRIPTOR	descriptor;

// The structure of an interrupt, call, trap or task gate descriptor
struct GATE
{
	ushort		offset0;		// Offset into segment, bit 0-15
	ushort		segment;		// Segment selector
	ushort		type;			// Some additional data
	ushort		offset1;		// Offset into segment, bit 16-31
} __attribute__((packed));

typedef struct GATE	gate;

// The structure of the task state segment
struct TSS
{
	ushort		back;			// Backlink field
	ushort		res0;			// all res* are reserved and set to 0
	ulong		esp0;			// Ring 0 ESP
	ushort		ss0;			// Ring 0 SS
	ushort		res1;
	ulong		esp1;			// Ring 1 ESP
	ushort		ss1;			// Ring 1 SS
	ushort		res2;
	ulong		esp2;			// Ring 2 ESP
	ushort		ss2;			// Ring 2 SS
	ushort		res3;
	ulong		cr3;			// CR3 (Page Direcory Base Register)

	ulong		eip,
				eflags,
				eax,
				ecx,
				edx,
				ebx,
				esp,
				ebp,
				esi,
				edi;

	ushort		es,
				res4,
				cs,
				res5,
				ss,
				res6,
				ds,
				res7,
				fs,
				res8,
				gs,
				res9;

	ushort		ldt;			// LDT descriptor
	ushort		res10;
	ushort		trap;			// bit 0 indicates trap-on-task-switch
	ushort		iomap;			// Offset of I/O Map into tss
} __attribute__((packed));

typedef	struct TSS	tss;


// A pseudo descriptor
struct PSEUDO
{
	ushort		limit;
	ulong		linear;			// Linear address
} __attribute__((packed));

typedef	struct PSEUDO	pseudo;


// A 16:32 far pointer
struct PWORD
{
	ulong		offset;
	ushort		selector;
} __attribute__((packed));

typedef struct PWORD	pword;

/** Function Prototypes **/
extern __inline__ boolean	IsLocal(ushort sel);
extern descriptor			*GetAddress(ushort sel);
extern ulong				GetBase(ushort sel);
extern ulong				GetLimit(ushort sel);
extern ushort				GetType(ushort sel);
extern boolean				BuildDesc(ushort sel, ulong base, ulong limit, ushort type);
extern void					BuildIDT(ushort nr, ushort sel, ulong offs, ushort type);
// extern void					GotoXY(ushort x, ushort y);
// extern void					kprint(const unsigned char *s);
// extern void					kprintf(const unsigned char *, ...);
extern void					CopyMem(ushort sseg, ulong src, ushort dseg, ulong dest, ulong n);
extern void					FillMem(ushort sel, ulong offs, byte b, ulong n);
// extern void					ClrScr(void);
// extern void					GotoXY(ushort x, ushort y);

// extern void					B2Str(char *buf, byte);
// extern void					W2Str(char *buf, ushort);
// extern void					L2Str(char *buf, ulong);

// extern __inline__ void		SetAttr(byte a);
// extern __inline__ byte		GetAttr(void);

extern ulong				Ptr2Lin(void *);
extern void 				*Lin2Ptr(ulong);



/** Inline Functions **/
static __inline__ byte inportb(ushort _port)
{
	byte rv;
	
	__asm__ __volatile__
	  ("inb %1,%0"
	  : "=a" (rv)
	  : "d" (_port));
	
	return rv;
}

static __inline__ byte inportb_p(ushort _port)
{
	byte rv;
	
	__asm__ __volatile__
	  ("inb %1,%0\n"
		"outb %%al, $0x80"
	  : "=a" (rv)
	  : "d" (_port)
	  );
	
	return rv;
}

static __inline__ ushort inportw(ushort _port)
{
	ushort rv;
	
	__asm__ __volatile__
	  ("inw %1,%0"
	  : "=a" (rv)
	  : "d" (_port));
	
	return rv;
}

static __inline__ ulong inportl(ushort _port)
{
	ulong rv;
	
	__asm__ __volatile__ 
	  ("inl %1,%0"
	  : "=a" (rv)
	  : "d" (_port));
	  
	return rv;
}

static __inline__ void outportb(ushort _port, byte _data)
{
	__asm__ __volatile__
	  ("outb %1,%0"
	  :
	  : "d" (_port), "a" (_data));
}

static __inline__ void outportb_p(ushort _port, byte _data)
{
	__asm__ __volatile__
	  ("outb %1,%0\n"
		"outb %%al, $0x80"
	  :
	  : "d" (_port), "a" (_data)
	  );
}

static __inline__ void outportw(ushort _port, ushort _data)
{
	__asm__ __volatile__
	  ("outw %1,%0"
	  :
	  : "d" (_port), "a" (_data));
}

static __inline__ void outportl(ushort _port, ulong _data)
{
	__asm__ __volatile__
	  ("outl %1,%0"
	  :
	  : "d" (_port), "a" (_data));
}

// These inline functions are here because you cannot pass #defines to __asm__
static __inline__ void SetDS(ushort selector)
{
	__asm__ __volatile__
	  ("movw %w0,%%ds"
	  :
	  : "rm" (selector));
}

static __inline__ ushort GetDS(void)
{
	ushort	result;
	__asm__ __volatile__
	  ("movw %%ds,%w0"
	  : "=rm" (result)
	  : );

	return result;
}

static __inline__ void SetES(ushort selector)
{
	__asm__ __volatile__
	  ("movw %w0,%%es"
	  :
	  : "rm" (selector));
}

static __inline__ ushort GetES(void)
{
	ushort	result;
	__asm__ __volatile__
	  ("movw %%es,%w0"
	  : "=rm" (result)
	  : );

	return result;
}

static __inline__ void SetFS(ushort selector)
{
	__asm__ __volatile__
	  ("movw %w0,%%fs"
	  :
	  : "rm" (selector));
}

static __inline__ ushort GetFS(void)
{
	ushort	result;
	__asm__ __volatile__
	  ("movw %%fs,%w0"
	  : "=rm" (result)
	  : );

	return result;
}

static __inline__ void SetGS(ushort selector)
{
	__asm__ __volatile__
	  ("movw %w0,%%gs"
	  :
	  : "rm" (selector));
}

static __inline__ ushort GetGS(void)
{
	ushort	result;
	__asm__ __volatile__
	  ("movw %%gs,%w0"
	  : "=rm" (result)
	  : );

	return result;
}

static __inline__ void PokeB(ushort sel, ulong offs, byte b)
{
	ushort		GS;
	
	GS = GetGS();
	
	SetGS(sel);
	__asm__ __volatile__
	  ("movb %b0,%%gs:(%k1);"
	  :
	  : "ri" (b), "r" (offs));
	 
	SetGS(GS);
}
	
static __inline__ void PokeW(ushort sel, ulong offs, ushort w)
{
	ushort		GS;
	
	GS = GetGS();
	
	SetGS(sel);
	__asm__ __volatile__
	  ("movw %w0,%%gs:(%k1);"
	  :
	  : "ri" (w), "r" (offs));
	 
	SetGS(GS);
}
	
static __inline__ void PokeL(ushort sel, ulong offs, ulong l)
{
	ushort		GS;
	
	GS = GetGS();
	
	SetGS(sel);
	__asm__ __volatile__
	  ("movl %k0,%%gs:(%k1);"
	  :
	  : "ri" (l), "r" (offs));
	 
	SetGS(GS);
}
	
static __inline__ byte PeekB(ushort sel, ulong offs)
{
	ushort		GS;
	byte		result;
	
	GS = GetGS();
	
	SetGS(sel);
	__asm__ __volatile__
	  ("movb %%gs:(%k1),%b0;"
	  : "=r" (result)
	  : "r" (offs));
	SetGS(GS);
	
	return result;
}

static __inline__ ushort PeekW(ushort sel, ulong offs)
{
	ushort		GS;
	ushort		result;
	
	GS = GetGS();
	
	SetGS(sel);
	__asm__ __volatile__
	  ("movw %%gs:(%k1),%w0;"
	  : "=r" (result)
	  : "r" (offs));
	SetGS(GS);
	
	return result;
}

static __inline__ ulong PeekL(ushort sel, ulong offs)
{
	ushort		GS;
	ulong		result;
	
	GS = GetGS();
	
	SetGS(sel);
	__asm__ __volatile__
	  ("movl %%gs:(%k1),%k0;"
	  : "=r" (result)
	  : "r" (offs));
	SetGS(GS);
	
	return result;
}
/*
static __inline__ void SetCurOffs(ushort offs)
{
	offs >>= 1;

	outportb(0x3d4, 0x0f);
	outportb(0x3d5, offs & 0xff);

	outportw(0x3d4, 0x0e);
	outportb(0x3d5, offs >> 8);
}
	
static __inline__ ushort GetCurOffs(void)
{
	ushort	offs;

	outportb(0x3d4, 0x0e);
	offs = inportb(0x3d5);
	
	offs <<= 8;
	
	outportb(0x3d4, 0x0f);
	offs += inportb(0x3d5);
	
	return offs << 1;
}

static __inline__ ushort CurX(void)
{
	return (GetCurOffs() >> 1) % 80;
}

static __inline__ ushort CurY(void)
{
	return (GetCurOffs() >> 1) / 80;
}

*/

#endif //__KERNEL_H

