/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel exception handler code.        ***/
/***    Copyright (C) 1997 by Joachim Breitsprecher                       ***/
/***       email: j.breitsprecher@schwaben.de                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***  Joachim Breitsprecher (HJB)                                         ***/
/***      <j.breitsprecher@schwaben.de>                                   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  22. Feb 1997  Coding started (HJB)                                  ***/
/****************************************************************************/
#include <kernel/selector.h>

#include <kernel/kernel.h>
#include <kernel/irq.h>
#include <kernel/timer.h>
#include <kernel/keyboard.h>
#include <kernel/schedule.h>
#include <kernel/memmgmt.h>
#include <kernel/except.h>
#include <kernel/kprint.h>
#include <kernel/console.h>
#include <stdarg.h>

static void		DefaultExceptionHandler(IntRegs *, int);
void			DoExc(IntRegs *, int);


const char	ExceptionMess[19][80] =
				{ "Exception 00: DIVISION BY ZERO OR DIVISION OVERFLOW\r\n",
				  "Exception 01: DEBUG EXCEPTION DETECTED\r\n",
				  "Exception 02: NON MASKABLE INTERRUPT\r\n",
				  "Exception 03: BREAKPOINT INSTRUCTION DETECTED\r\n",
				  "Exception 04: INTO DETECTED OVERFLOW\r\n",
				  "Exception 05: BOUND RANGE EXCEEDED\r\n",
				  "Exception 06: INVALID OPCODE\r\n",
				  "Exception 07: PROCESSOR EXTENSION NOT AVAILABLE\r\n",
				  "Exception 08: DOUBLE FAULT DETECTED\r\n",
				  "Exception 09: PROCESSOR EXTENSION PROTECTION FAULT\r\n",
				  "Exception 0A: INVALID TASK STATE SEGMENT\r\n",
				  "Exception 0B: SEGMENT NOT PRESENT\r\n",
				  "Exception 0C: STACK FAULT\r\n",
				  "Exception 0D: GENERAL PROTECTION FAULT\r\n",
				  "Exception 0E: PAGE FAULT\r\n",
				  "Exception 0F: UNKNOWN EXCEPTION\r\n",
				  "Exception 10: COPROCESSOR ERROR\r\n",
				  "Exception 11: ALIGNMENT ERROR\r\n",
				  "Exception 12: MACHINE CHECK EXCEPTION\r\n"};

void	(*ExcHandler)(IntRegs *, int);


#define EXC_STUB_NAME2(nr) nr##_stub(void)
#define EXC_STUB_NAME(nr) EXC_STUB_NAME2(Exception##nr)

#define EXC_STUB(Nr)                      \
void EXC_STUB_NAME(Nr);                   \
__asm__ (                                 \
        "\n.align 4\n"                    \
        "_Exception" #Nr "_stub:\n\t"     \
        "push   $0\n\t"                   \
        "pushal\n\t"                      \
        "pushl %ds\n\t"                   \
        "pushl %ss\n\t"                   \
        "pushl %es\n\t"                   \
        "pushl %fs\n\t"                   \
        "pushl %gs\n\t"                   \
	    "pushl  $" #Nr "\n\t"             \
        "lea   4(%esp), %EAX\n\t"         \
        "pushl %eax\n\t"                  \
        "call   _DoExc\n\t"               \
	    "addl   $8,%esp\n\t"              \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popal\n\t"                       \
        "popl   %eax\n\t"                 \
        "iret\n"                          \
       );

#define EXC_STUB_ERR(Nr)                  \
void EXC_STUB_NAME(Nr);                   \
__asm__ (                                 \
        "\n.align 4\n"                    \
        "_Exception" #Nr "_stub:\n\t"     \
        "pushal\n\t"                      \
        "pushl %ds\n\t"                   \
        "pushl %ss\n\t"                   \
        "pushl %es\n\t"                   \
        "pushl %fs\n\t"                   \
        "pushl %gs\n\t"                   \
	    "pushl $" #Nr "\n\t"              \
        "lea   4(%esp), %EAX\n\t"         \
        "pushl %eax\n\t"                  \
        "call   _DoExc\n\t"               \
        "addl   $8,%esp\n\t"              \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popl   %eax\n\t"                 \
        "popal\n\t"                       \
        "popl   %eax\n\t"                 \
        "iret\n"                          \
       );


EXC_STUB(0);
EXC_STUB(1);
EXC_STUB(2);
EXC_STUB(3);
EXC_STUB(4);
EXC_STUB(5);
EXC_STUB(6);
EXC_STUB(7);
EXC_STUB_ERR(8);
EXC_STUB(9);
EXC_STUB_ERR(10);
EXC_STUB_ERR(11);
EXC_STUB_ERR(12);
EXC_STUB_ERR(13);
EXC_STUB_ERR(14);
EXC_STUB(15);
EXC_STUB(16);
EXC_STUB_ERR(17);
EXC_STUB_ERR(18);

void DoExc(IntRegs *regs, int nr)
{
	(*ExcHandler)(regs, nr);
}

static void DefaultExceptionHandler(IntRegs *r, int nr)
{
	ulong		CR2;

	__asm__ __volatile__
	  ("movl %%cr2,%0"
	  :"=r" (CR2)
	  : );

	SetAttr(0x0F);
	kprint(ExceptionMess[nr]);

	SetAttr(0x07);
	kprintf("Error Code = %08lX, CR2 = %08lX\r\n", r->ErrorCode, CR2);
	kprintf("EAX = %08lX, EBX = %08lX, ECX = %08lX, EDX = %08lX, EFL = %08lX\r\n", r->EAX, r->EBX, r->ECX, r->EDX, r->EFLAGS);
	kprintf("ESI = %08lX, EDI = %08lX, EBP = %08lX, ESP = %08lX, EIP = %08lX\r\n", r->ESI, r->EDI, r->EBP, r->ESP, r->EIP);

	kprintf("CS = %04X: ", r->CS);
	DumpDesc(r->CS);
	kprintf("DS = %04X: ", r->DS);
	DumpDesc(r->DS);
	kprintf("ES = %04X: ", r->ES);
	DumpDesc(r->ES);
	kprintf("FS = %04X: ", r->FS);
	DumpDesc(r->FS);
	kprintf("GS = %04X: ", r->GS);
	DumpDesc(r->GS);
	kprintf("SS = %04X: ", r->SS);
	DumpDesc(r->SS);

	// Halt machine
	for(;;) __asm__ __volatile__ ("cli; hlt");
}

/****************************************************************************/
/*** TODO: Add support for LDT of crashed task; for now only the kernel's ***/
/***       LDT is used for all local descriptors!                         ***/
/****************************************************************************/
/*** NOTE: The exception handlers with error code (08, 0A-0E, 11, 12)     ***/
/***       have a "stub" handler in except.s to get the error code        ***/
/****************************************************************************/
// This function dumps all properties of a descriptor to the screen
void DumpDesc(ushort sel)
{
	char	buf[9];
	ushort	type;
	
	kprintf("Base: %08lx, Limit: %08lx, Type: %03x", GetBase(sel), GetLimit(sel), (type=GetType(sel)));
	
	if(type & 0x800)
		kprint(" G");
	else
		kprint(" -");
		
	if(type & 0x008)
	{
		if(type & 0x400)
			kprint("D-");
		else
			kprint("--");
	}
	else
	{
		if(type & 0x400)
			kprint("B-");
		else
			kprint("--");
	}
		
	if(type & 0x100)
		kprint("1");
	else
		kprint("0");
		
	if(type & 0x080)
		kprint("P");
	else
		kprint("-");
		
	buf[0] = (type & 0x060) >> 5;
	buf[0] += '0';
	buf[1] = '\0';
	kprint(buf);
	
	if(type & 0x010)
		kprint("S");
	else
		kprint("-");
		
	if(type & 0x008)
	{
		kprint("X");
		if(type & 0x004)
			kprint("C");
		else
			kprint("-");
			
		if(type & 0x002)
			kprint("R");
		else
			kprint("-");
	}
	else
	{
		kprint("D");
		if(type & 0x004)
			kprint("\x19");
		else
			kprint("-");
			
		if(type & 0x002)
			kprint("W");
		else
			kprint("R");
	}
	
	if(type & 0x001)
		kprint("A\r\n");
	else
		kprint("-\r\n");
}		
		
////////////////////////////////////////////////////////////////////////////
// Public Functions

void SetExceptionHook(void (*fnc)(IntRegs *, int))
{
	ExcHandler = fnc;
}

void ResetExceptionHook(void)
{
	ExcHandler = DefaultExceptionHandler;
}

void InitExceptions(void)
{
	BuildIDT(0, SEL_CKERNEL, (ulong)Exception0_stub, 0x8E00);
	BuildIDT(1, SEL_CKERNEL, (ulong)Exception1_stub, 0x8E00);
	BuildIDT(2, SEL_CKERNEL, (ulong)Exception2_stub, 0x8E00);
	BuildIDT(3, SEL_CKERNEL, (ulong)Exception3_stub, 0x8E00);
	BuildIDT(4, SEL_CKERNEL, (ulong)Exception4_stub, 0x8E00);
	BuildIDT(5, SEL_CKERNEL, (ulong)Exception5_stub, 0x8E00);
	BuildIDT(6, SEL_CKERNEL, (ulong)Exception6_stub, 0x8E00);
	BuildIDT(7, SEL_CKERNEL, (ulong)Exception7_stub, 0x8E00);
	BuildIDT(8, SEL_CKERNEL, (ulong)Exception8_stub, 0x8E00);
	BuildIDT(9, SEL_CKERNEL, (ulong)Exception9_stub, 0x8E00);
	BuildIDT(10, SEL_CKERNEL, (ulong)Exception10_stub, 0x8E00);
	BuildIDT(11, SEL_CKERNEL, (ulong)Exception11_stub, 0x8E00);
	BuildIDT(12, SEL_CKERNEL, (ulong)Exception12_stub, 0x8E00);
	BuildIDT(13, SEL_CKERNEL, (ulong)Exception13_stub, 0x8E00);
	BuildIDT(14, SEL_CKERNEL, (ulong)Exception14_stub, 0x8E00);
	BuildIDT(15, SEL_CKERNEL, (ulong)Exception15_stub, 0x8E00);
	BuildIDT(16, SEL_CKERNEL, (ulong)Exception16_stub, 0x8E00);
	BuildIDT(17, SEL_CKERNEL, (ulong)Exception17_stub, 0x8E00);
	BuildIDT(18, SEL_CKERNEL, (ulong)Exception18_stub, 0x8E00);

	SetExceptionHook(DefaultExceptionHandler);
}

