/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel thread management code.        ***/
/***    Copyright (C) 1997 by Martin Kortmann                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  09. mar 1997  Coding started (MK)                                   ***/
/****************************************************************************/
#include <kernel/selector.h>
#include <kernel/kernel.h>
#include <kernel/irq.h>
#include <kernel/timer.h>
#include <kernel/keyboard.h>
#include <kernel/schedule.h>
#include <kernel/memmgmt.h>
#include <kernel/kprint.h>
#include <kernel/console.h>

extern tss     kerneltss;
#define K_STACK_MAGIC 0x12344321	// stack magic number
extern long    KernelStack[];

extern ulong   Ptr2Lin(void *ptr);
extern boolean BuildDesc(ushort sel, ulong base, ulong limit, ushort type);

static ulong LastPid;
static Task IdleTask;
static Task *CurrentTask;

static Task KernelTasks[NR_OF_KERNEL_TASKS];

static volatile int ReSchedule;

static void SwitchToTask (Task *NewTask)
{
   if (NewTask == CurrentTask)
      return;

   if (NewTask -> Counter <= 0)
      NewTask -> Counter = NewTask -> Priority;
   CurrentTask = NewTask;

   __asm__ __volatile__ (
     "ljmp %0" : : "m" (*(((char *)&NewTask->TSSSel)-4)) : "memory"
   );
}

static void DoNewTask (void)
{
   void (*fnc)(void) = 0;
	long flags;

   if (KernelStack[0] != K_STACK_MAGIC)
   {
   	  __asm__ __volatile__ ("cli");
	  kprint ("\r\n !!! KERNEL STACK CORRUPT !!! \r\n");
	  kprint ("\r\n !!! KERNEL STACK CORRUPT !!! \r\n");
	  for (;;)
		 ;
   }

   save_flags (flags);
   cli ();

   fnc = CurrentTask -> StartFnc;
   CurrentTask -> StartFnc = 0;

   restore_flags (flags);

   fnc ();

   CurrentTask -> State = TASKFREE;

   Schedule ();
}

//////////////////////////////////////////////////////////////////////////
// public Funktions

int InIdleTask (void)
{
   return (CurrentTask == &IdleTask);
}

int TimeCurrentTask (void)
{
 	int  i;
	long ticks = GetTickCount();

   if (! CurrentTask)
      return(0);

   for (i = 0; i < NR_OF_KERNEL_TASKS; i++)
   {
      if (KernelTasks[i].State == TASKSLEEPING)
      {
		 	if (ticks >= KernelTasks[i].TimeToWakeup)
			{
				KernelTasks[i].State = TASKREADY;
			   ReSchedule = 1;
			}
      }
   }

   if (CurrentTask -> Counter > 0)
      if ((-- CurrentTask -> Counter) > 0)
         return (ReSchedule);

   ReSchedule = 1;

   return (1);
}

int CreateKernelTask (void (*fnc)(void), int Priority)
{
   int i;
	long flags;
   Task * newTask = 0;

   if (! fnc)
	{
      kprintf ("KERNEL: Warning, 0 functon in CreateTask!\r\n");
      return (0);
   }

   if (Priority < -9)
      Priority = -9;
   else if (Priority > 9)
      Priority = 9;

   for (i = 0; i < NR_OF_KERNEL_TASKS; i++)
   {
      if (KernelTasks[i].State == TASKFREE)
      {
         newTask = KernelTasks +i;
         break;
      }
   }

   if (! newTask)
   {
      kprintf ("KERNEL: Warning, running out of Kernel tasks!\r\n");
      return (-1);
   }

	if (! CurrentTask)
	{
      kprintf ("KERNEL: Task system not initialized!\r\n");
      return (0);
	}

   save_flags (flags);
	cli();

   *newTask = *CurrentTask;
   newTask->State         = TASKWAITING;

   restore_flags (flags);

   newTask->pid           = ++LastPid;
   newTask->lasteip       = (ulong) DoNewTask;
   newTask->Counter       = newTask->Counter >> 2;
   newTask->Priority     += Priority;
   newTask->StartFnc      = fnc;
   newTask->TimeInTask    = 0;
   newTask->TimeInKernel  = 0;

   newTask->TaskTSS.esp   = (long) GetMemoryPage();
   newTask->TaskTSS.esp   =
   newTask->TaskTSS.esp0  = newTask->TaskTSS.esp + 4096;

	newTask->TaskTSS.eflags= 0x202;
   newTask->TaskTSS.eip   = (long) DoNewTask;
   newTask->TaskTSS.iomap = 0xFFFF;
   newTask->TSSSel        = 0x050 + (8 * i);

	BuildDesc(newTask->TSSSel, Ptr2Lin(&(newTask->TaskTSS)), sizeof(tss), 0x089);

   newTask->State         = TASKREADY;

   return (i);
}

// nchsten Task schedulen
void Schedule (void)
{
   Task       *NextTask = 0;
   Task       *task;
   int         highestPriority = -1;
   int         i;

   if (! CurrentTask)
      return;

   ReSchedule = 0;

   // Alle nicht mehr wartenden Tasks aufwecken
   for (i = 0, task = KernelTasks; i < NR_OF_KERNEL_TASKS; i++, task++)
      if (task->State == TASKNOTWAITING)
         task->State = TASKREADY;

   // den Task mit der grten Prioritt suchen
   for (i = 0, task = KernelTasks; i < NR_OF_KERNEL_TASKS; i++, task++)
   {
      if (task->State == TASKREADY)
      {
         if (task->Counter > highestPriority)
         {
            highestPriority = task->Counter;
            NextTask = task;
         }
      }
   }

   // falls keiner lauffhig ist, alle Task in der Prioritt 'hochziehen'
   if (highestPriority <= 0)
      for (i = 0, task = KernelTasks; i < NR_OF_KERNEL_TASKS; i++, task++)
             task->Counter = (task->Counter >> 2) + task->Priority;

   if (NextTask)
      SwitchToTask (NextTask);
   else
      SwitchToTask (&IdleTask);
}

void InitKernelTasking (void)
{
   int i;

   LastPid = 0;
   ReSchedule = 0;

   for (i = 0; i < NR_OF_KERNEL_TASKS; i++)
      KernelTasks[i].State = TASKFREE;

   // Setup Idle Task.
   // this is also the starting kenerl task
   IdleTask.pid           = ++LastPid;
   IdleTask.lasteip       = 0;
   IdleTask.State         = TASKREADY;
   IdleTask.Counter       = 0;
   IdleTask.Priority      = 10;
   IdleTask.TaskTSS       = kerneltss;
   IdleTask.TimeInTask    = 0;
   IdleTask.TimeInKernel  = 0;
   IdleTask.CurrentConsole= 0;

   IdleTask.TSSSel        = SEL_KERNELTSS;

   CurrentTask = &IdleTask;
}

Task *GetCurrentTask (void)
{
   if (InIdleTask())
      return (0);

   return (CurrentTask);
}

int NeedSchedule (void)
{
   return (ReSchedule);
}

void RequestReschedule (void)
{
   ReSchedule = 1;
}

void Sleep (int ms)
{
	Task *t = CurrentTask;

 	if (! t || t == &IdleTask)
	  return;

	t -> TimeToWakeup = GetTickCount() + ms / 10;
	t -> State =  TASKSLEEPING;

	Schedule ();
}

void DumpTaskStatistic (void)
{
   int i;
   long flags;

   save_flags (flags);
   cli ();

   GotoXY(0, 14);
   kprintf ("Task Statistik:\r\npid  State           Count   Pri.   EIP    \r\n");

   for (i = 0; i < NR_OF_KERNEL_TASKS; i++)
   {
      if (KernelTasks[i].State == TASKFREE)
         continue;

      kprintf ("%2d   ", KernelTasks[i].pid);

      if (KernelTasks[i].State == TASKREADY)
         kprintf ("RUNNING         ");
      else if (KernelTasks[i].State == TASKWAITING)
         kprintf ("WAITING         ");
      else if (KernelTasks[i].State == TASKNOTWAITING)
         kprintf ("NOT WAITING     ");
      else if (KernelTasks[i].State == TASKSLEEPING)
         kprintf ("SLEEPING        ");
      else
         kprintf ("--UNKNOWN--     ");

      kprintf ("%-8d", KernelTasks[i].Counter);
      kprintf ("%-8d", KernelTasks[i].Priority);
	   kprintf ("%08X", KernelTasks[i].lasteip);

      kprintf ("     \r\n");
   }
   for (; i < 6; i++)
      kprintf ("                                            \r\n");

   restore_flags (flags);
}


