/****************************************************************************/
/*** This is the Freedows '98 Cache Kernel startup code.                  ***/
/*** It is called after the loader has switched to Protected Mode.        ***/
/*** It will initialize the basic memory management (page tables, task    ***/
/*** state segments etc)                                                  ***/
/***    Copyright (C) 1997 by Joachim Breitsprecher                       ***/
/***       email: j.breitsprecher@schwaben.de                             ***/
/***                                                                      ***/
/***    This file is part of the Freedows '98 Project                     ***/
/****************************************************************************/
/*** Contributors: (If you modify this, please put your name/email here   ***/
/***  Joachim Breitsprecher (HJB)                                         ***/
/***      <j.breitsprecher@schwaben.de>                                   ***/
/***                                                                      ***/
/*** File History: (Please record any changes here)                       ***/
/***  22. Feb 1997  Coding started (HJB)                                  ***/
/****************************************************************************/
#include <kernel/selector.h>

#include <kernel/kernel.h>
#include <kernel/console.h>
#include <kernel/irq.h>
#include <kernel/timer.h>
#include <kernel/keyboard.h>
#include <kernel/schedule.h>
#include <kernel/kprint.h>
#include <kernel/memmgmt.h>
#include <kernel/except.h>
#include <kernel/lpt.h>

#ifdef _INCLUDE_DEBUGGER_
#include "../debugger/debugger.h"
#endif

extern void InitFloppy  (void);
extern void InitRamdisk (void);
extern void	InitMiniFS  (void);
extern void LoadRamdisk (void);
extern volatile int RamdiskLoaded;

/****************************************************************************/
/*** Some global kernel data                                              ***/
/****************************************************************************/
extern descriptor *gdt;			// This will be the real GDT
extern descriptor *ldt;	 		// The kernel LDT
extern gate		  *idt;	 		// Our Interrupt Descriptor Table
extern ulong	KernelBase;		// Base address of the kernel segment

									// The Kernel stack
#define	K_STACK_SIZE  1024			// size of Stack in dwords
#define	K_STACK_MAGIC 0x12344321	// stack magic number
long KernelStack [K_STACK_SIZE];
									// and a pointer to it
struct
{
	long * a;
	short b;
} KernelStackStart = { & KernelStack[K_STACK_SIZE],	SEL_DKERNEL	};

tss			ExcTask[19];	// Task state segments for exception handlers
tss			DummyTSS;		// Dummy TSS for exception return
ulong			ExcStack[256];	// Auxillary stack for exception handlers

// A function to build a descriptor in the GDT
boolean	BuildGDT(ushort	nr,	ulong base,	ulong limit, ushort	type)
{
	descriptor	*d = gdt+nr;

	if(type	& 0x0800)			// if segment is page granular
		limit >>= 12;			// calculate number of pages from limit
	else						// if it is byte granular
		if(limit>=0x00100000)	// is limit greater than 1 MB?
			return FALSE;		// yes, -> return with error

	d->limit0 =	(ushort)limit;
	d->limit1 =	(ushort)(limit >> 16) |	((type >> 4) & 0xf0);
	d->base0 = (ushort)base;
	d->base1 = (byte)(base >> 16);
	d->base2 = (byte)(base >> 24);
	d->type	= (byte)type;

	return TRUE;
}

// A function to build a descriptor in the LDT
boolean	BuildLDT(ushort	nr,	ulong base,	ulong limit, ushort	type)
{
	descriptor	*d = ldt+nr;

	if(type	& 0x0800)			// if segment is page granular
		limit >>= 12;			// calculate number of pages from limit
	else						// if it is byte granular
		if(limit>=0x00100000)	// is limit greater than 1 MB?
			return FALSE;		// yes, -> return with error

	d->limit0 =	(ushort)limit;
	d->limit1 =	(ushort)(limit >> 16) |	((type >> 4) & 0xf0);
	d->base0 = (ushort)base;
	d->base1 = (byte)(base >> 16);
	d->base2 = (byte)(base >> 24);
	d->type	= (byte)type;

	return TRUE;
}

// The next functions return different properties of a descriptor
//    Global or Local Descriptor?
__inline__ boolean IsLocal(ushort sel)
{
	return (sel&0x04 ? TRUE	: FALSE);
}

descriptor *GetAddress(ushort sel)
{
	if(IsLocal(sel))
		return ldt+(sel>>3);
	else
		return gdt+(sel>>3);
}

//    Base Address
ulong GetBase(ushort sel)
{
	descriptor	*d;
	ulong		base;
	
	d =	GetAddress(sel);
	
	base = d->base0;
	base +=	d->base1<<16;
	base +=	d->base2<<24;
	
	return base;
}

//    Segment Limit
ulong GetLimit(ushort sel)
{
	descriptor	*d;
	ulong		limit;
		
	d =	GetAddress(sel);
	
	limit =	d->limit0;
	limit += (d->limit1	& 0x0f)	<< 16;
	
	if(d->limit1 & 0x80)
	{
		limit <<= 12;
		limit += 0x0fff;
	}

	return limit;
}

//    Descriptor Type (Gran, Opsize: high byte; pres, dpl, sys, type: low byte
ushort GetType(ushort sel)
{
	descriptor	*d;
	ushort		type;
	
	d =	GetAddress(sel);
	
	type = d->type;
	type +=	(d->limit1 & 0xf0) << 4;
	
	return type;
}

// This function builds a descriptor in GDT or LDT, depending on the TI bit
boolean	BuildDesc(ushort sel, ulong	base, ulong	limit, ushort type)
{
	if(IsLocal(sel))
		return BuildLDT(sel>>3,	base, limit, type);
	else
		return BuildGDT(sel>>3,	base, limit, type);
}

// This function builds a gate in the IDT
void BuildIDT(ushort nr, ushort	sel, ulong offs, ushort	type)
{
	idt[nr].segment	= sel;
	idt[nr].type = type;
	idt[nr].offset0	= (ushort)offs;
	idt[nr].offset1	= offs >> 16;
}

// CopyMem - Allows to copy memory with "far pointer" access
void CopyMem(ushort	sseg, ulong	src, ushort	dseg, ulong	dest, ulong	n)
{
	ushort		DS,	ES;			// To save ds and es
	
	DS = GetDS();
	ES = GetES();

	SetDS(sseg);
	SetES(dseg);
	__asm__	__volatile__
	  ("cld;"
	   "movl %k0,%%ecx;"
	   "movl %k1,%%esi;"
	   "movl %k2,%%edi;"
	   "shrl $2,%%ecx;"
	   "rep;movsl;"
	   "movl %k0,%%ecx;"
	   "andl $3,%%ecx;"
	   "rep;movsb;"
	  :
	  :	"rm" (n), "rm" (src), "rm" (dest)
	  :	"%ecx",	"%esi",	"%edi"
	 );
	  
	SetDS(DS);
	SetES(ES);
}

// FillMem - Fills a memory area with the specified byte
void FillMem(ushort	sel, ulong offs, byte b, ulong n)
{
	ushort		ES;
	
	ES=GetES();
	
	SetES(sel);
	__asm__	__volatile__
	  ("movl %k0,%%ecx;"
	   "movl %k1,%%edi;"
	   "movb %b2,%%al;"
	   "rep;stosb;"
	  :
	  :	"rm" (n), "rm" (offs), "rm"	(b)
	 : "%ecx", "%edi", "%al"
	  );
	 
	SetES(ES);
}


// ClrScr - clears the screen (w/ current attribute) and sets cursor to 0,0
static void ClrBootScr(void)
{
	int		i;
	ushort	c = (7 << 8)	+ ' ';

 	for(i =	0; i < 80*50; i += 2)
		PokeW(SEL_DLINEAR, 0xb8000 + i, c);

	outportb(0x3d4, 0x0f);
	outportb(0x3d5, 0);

	outportw(0x3d4, 0x0e);
	outportb(0x3d5, 0);
}

ulong Ptr2Lin(void *ptr)
{
	return ((ulong)ptr)	+ KernelBase;
}

void *Lin2Ptr(ulong	lin)
{
	return (void *)(lin	- KernelBase);
}

void PrintSystemStatistiks (void)
{
   unsigned	long Ticks = GetTickCount ();

   GotoXY(0, 14);
   kprint ("System statistics:\r\n");
   kprintf ("System is up %d ticks (x 10ms) = %d Seconds\r\n", Ticks, Ticks/100);
   kprint ("Number of interrupts:\r\n");
   DumpIRQStatistic	();
}

/****************************************************************************/
// This tasks running forever...
void TestTask1	(void)
{
   static int TaskNo = 0;
   int	No = 0;
   int xpos	= TaskNo * 20;
	long flags;

	GetCurrentTask() -> CurrentConsole = 2;

   GotoXY(xpos , 24);
   kprintf("Task %d:", ++TaskNo);

   xpos	+= 8;

	Sleep (1);

   for (;;)
   {
	  save_flags (flags);
	  cli();

	  GotoXY (xpos,	24);
	  kprintf ("%08X", ++No);

	  restore_flags(flags);

	}
}

// This task waits for keystrokes
void TestTask2	(void)
{
   int Key;
	long flags;

	Sleep (1);

	for	(;;)
	{
	  Key =	getch ();

	  save_flags (flags);
	  cli();

	  GotoXY (0, 22);
	  kprintf ("Got a Key: '%c'", Key);

	  restore_flags(flags);

	  if (Key == 'b')
	 	 __asm__ __volatile__ ("int $3");
	  else if (Key == 'z')	// Nullpointer access
	 	 * ((char *) 0)	= 0;;
	}
}

/****************************************************************************/
/*** This is the entry point. The loader's GoPM function will jump here   ***/
/*** after switching to protected mode                                    ***/
/****************************************************************************/
/*** Note that I did not use the standard entry symbol "main", since the  ***/
/*** compiler seems to put some special introduction at the beginning of  ***/
/*** any function called "main", which involves a call to "__main"...     ***/
/****************************************************************************/
/*** The segment registers can be assumed to contain the following:       ***/
/*** CS         - 32 bit 4GB segment with base address = kernel base      ***/
/*** DS,ES      - 4GB (big) data segment with base address = kernel base  ***/
/*** FS,GS      - 4GB (big) data segment with base address 0x00000000     ***/
/***                                                                      ***/
/*** You can further assume that we are still in the first MB of RAM      ***/
/*** and the BSS Segment was cleared.                                     ***/
/****************************************************************************/

__asm__	(
		 ".globl start        				\n\t"
		 "start:              				\n\t"

		 // clear BSS section
		 "xorl %eax,%eax      				\n\t"
		 "movl $_edata,%edi   				\n\t"
		 "movl $_end,%ecx     				\n\t"
		 "subl %edi,%ecx      				\n\t"
		 "cld                 				\n\t"
		 "rep                 				\n\t"
		 "stosb               				\n\t"

		 // setup kernel stack
		 "lss  _KernelStackStart, %esp		\n\t"

		 // and jump to main
		 "jmp _Main           				\n\t"

		 // we will never return...
		);

long No;
long flags;
void Main(void)
{
   // Setup kernel Stack
   KernelStack[0] =	K_STACK_MAGIC;

	// Clear screen
   ClrBootScr ();

	// Initialize the kernel systems
   InitMemory		 	();

   InitExceptions	 	();
   InitIRQ			 	(); // Init IRQ System, note: this also enables interrupts!
   InitTimer		 	();
   InitKernelTasking ();
   InitKeyboard	 	();
	InitConsole		 	();

#ifdef _INCLUDE_DEBUGGER_
	// setup the debugging system
	InitDebugger		();
#endif

	InitFloppy		 	();
	InitLPT			 	();
	InitRamdisk     	();

	// steup a few hotkeys
   SetHotkey (0x0044, KBD_SHIFT, PrintSystemStatistiks);
   SetHotkey (0x0044, KBD_ALT  , DumpMemoryStatistic);
   SetHotkey (0x0044, KBD_CTRL,  DumpTaskStatistic);

	CreateKernelTask   (LoadRamdisk, 0);

	// wait for the ramdisk to be loaded...
	for (;;)
	{
		if (RamdiskLoaded != 0)
		  	break;
	}

	InitMiniFS ();

	kprint("\nFreedows '98 is successfully started!\r\nPlease reboot now...\n");

	// create a few tasks
	CreateKernelTask   (TestTask1, -5);
	CreateKernelTask   (TestTask1, -5);
	CreateKernelTask   (TestTask1, -5);
	CreateKernelTask   (TestTask1, -5);
	CreateKernelTask   (TestTask2, -5);

	No = 0;
   // Idle loop:
	for(;;)
   {
	  save_flags (flags);
	  cli();

	  GotoXY (0, 23);
	  kprintf ("Enter Idle : %08X", ++No);

	  restore_flags(flags);

		__asm__	__volatile__ ("hlt");
   }

   // never reached
}

